﻿/*
Copyright (c) 2014 Stephen Stair (sgstair@akkit.org)
Additional code Miguel Parra (miguelvp@msn.com)
Port to single File and change/add something Joe-c (www.joe-c.de)

Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
*/

#region Usings
using System.ComponentModel;
using System.Runtime.InteropServices;
using Microsoft.Win32.SafeHandles;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Text.RegularExpressions;
using System.Drawing;
using System.Drawing.Imaging;
//using System.Drawing.Drawing2D;
#endregion

namespace SeekThermal
{//http://www.pcrpr.com/reggetvaluea-advapi32-dll.php
	public class ThermalFrame
    {
        public readonly int Width=206, Height=156;
        public readonly byte[] RawDataBytes;
        public ushort[,] Data;
        public ushort MinValue=65535;
        public ushort MaxValue=0;
        public ushort AvrValue=0;
        public ushort[] LineCal= new ushort[156];
        public readonly byte StatusByte;
        
        public readonly UInt16[] RawDataU16;
//		public bool[,] BadpixelMap=new bool[206,156];
        
        internal ThermalFrame(Byte[] data)
        {
            // Original data stream.
            if (data==null) { return; }
            RawDataBytes = data;
			
            StatusByte = data[20]; 
            //The first frames have the following IDs:
			//frame 1: 4	frame 5: 10
			//frame 2: 9	frame 6: 5
			//frame 3: 8	frame 7: 1
			//frame 4: 7	frame 8: 3
			//So, after the initialization sequence, you'll get something like this:
			//6, 1, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 3 etc.
			Data = new ushort[Width,Height];
			UInt32 AVR=0; ushort cnt=0; int x=0,y=0;
			int HexagonCnt = 10;
			for (int i=0;i<RawDataBytes.Length ;i+=2 ) {
				ushort val = (ushort)(data[i+1]<<8|data[i]);
				if (i==HexagonCnt) {
					HexagonCnt+=15;
				} else {
					if (x<Width) {
						if (val>2000&&val<12000) {
							AVR+=val; cnt++;
	                		Data[x,y] = (ushort)val;
						}
					} else if (x==206) {
						//Line Cal Pixel
						LineCal[y] = (ushort)((float)val/10f);
					}
				}
                x++;
                if (x==Width+2) {
                	y++;
                	x=0;
                	if (y==Height) {
                		break;
                	}
                }
			}
			if (cnt>0) {
				//cnt+=5000;
				AvrValue=(ushort)(AVR/cnt);
				//AvrValue=(ushort)(AvrValue*4);
			}
        }
        
        public void Frame_RemoveDeathPixel(bool[,] DPMap)
        {
        	//1  2  3
			//4  px 6
			//7  8  9
        	//Mitte ####################
			for (int y = 1; y < 155; ++y) {
				for (int x = 1; x < 205; ++x) {
        			if (DPMap[x,y]) { continue; } //Pixel ist valid
					long Val=0; byte Cnt=0;
					if (DPMap[x-1,y-1]) { Val+=Data[x-1,y-1]; Cnt++; } //1
					if (DPMap[x,y-1]) { Val+=Data[x,y-1]; Cnt++; } //2
					if (DPMap[x+1,y-1]) { Val+=Data[x+1,y-1]; Cnt++; } //3
					if (DPMap[x-1,y]) { Val+=Data[x-1,y]; Cnt++; } //4
					if (DPMap[x+1,y]) { Val+=Data[x+1,y]; Cnt++; } //6
					if (DPMap[x-1,y+1]) { Val+=Data[x-1,y+1]; Cnt++; } //7
					if (DPMap[x,y+1]) { Val+=Data[x,y+1]; Cnt++; } //8
					if (DPMap[x+1,y+1]) { Val+=Data[x+1,y+1]; Cnt++; } //9
					if (Cnt!=0) { Data[x,y]=(ushort)(Val/Cnt); }
				}
			}
        	//Rand Links ####################
			for (int y = 1; y < 155; ++y) {
    			if (DPMap[0,y]) { continue; } //Pixel ist valid
				long Val=0; byte Cnt=0;
				if (DPMap[0,y-1]) { Val+=Data[0,y-1]; Cnt++; } //2
				if (DPMap[1,y-1]) { Val+=Data[1,y-1]; Cnt++; } //3
				if (DPMap[1,y]) { Val+=Data[1,y]; Cnt++; } //6
				if (DPMap[0,y+1]) { Val+=Data[0,y+1]; Cnt++; } //8
				if (DPMap[1,y+1]) { Val+=Data[1,y+1]; Cnt++; } //9
				if (Cnt!=0) { Data[0,y]=(ushort)(Val/Cnt); }
			}
        	//Rand Rechts ####################
			for (int y = 1; y < 155; ++y) {
    			if (DPMap[205,y]) { continue; } //Pixel ist valid
				long Val=0; byte Cnt=0;
				if (DPMap[204,y-1]) { Val+=Data[204,y-1]; Cnt++; } //1
				if (DPMap[205,y-1]) { Val+=Data[205,y-1]; Cnt++; } //2
				if (DPMap[204,y]) { Val+=Data[204,y]; Cnt++; } //4
				if (DPMap[204,y+1]) { Val+=Data[204,y+1]; Cnt++; } //7
				if (DPMap[205,y+1]) { Val+=Data[205,y+1]; Cnt++; } //8
				if (Cnt!=0) { Data[205,y]=(ushort)(Val/Cnt); }
			}
        	//Rand Oben ####################
			for (int x = 1; x < 205; ++x) {
				if (DPMap[x,0]) { continue; } //Pixel ist valid
				long Val=0; byte Cnt=0;
				if (DPMap[x-1,0]) { Val+=Data[x-1,0]; Cnt++; } //4
				if (DPMap[x+1,0]) { Val+=Data[x+1,0]; Cnt++; } //6
				if (DPMap[x-1,1]) { Val+=Data[x-1,1]; Cnt++; } //7
				if (DPMap[x,1]) { Val+=Data[x,1]; Cnt++; } //8
				if (DPMap[x+1,1]) { Val+=Data[x+1,1]; Cnt++; } //9
				if (Cnt!=0) { Data[x,0]=(ushort)(Val/Cnt); }
			}
        	//Rand Unten ####################
			for (int x = 1; x < 205; ++x) {
				if (DPMap[x,155]) { continue; } //Pixel ist valid
				long Val=0; byte Cnt=0;
				if (DPMap[x-1,154]) { Val+=Data[x-1,154]; Cnt++; } //1
				if (DPMap[x,154]) { Val+=Data[x,154]; Cnt++; } //2
				if (DPMap[x+1,154]) { Val+=Data[x+1,154]; Cnt++; } //3
				if (DPMap[x-1,155]) { Val+=Data[x-1,155]; Cnt++; } //4
				if (DPMap[x+1,155]) { Val+=Data[x+1,155]; Cnt++; } //6
				if (Cnt!=0) { Data[x,155]=(ushort)(Val/Cnt); }
			}
        	//Eckenpixel ################
        	//oben links
        	if (!DPMap[0,0]) {
        		long Val=0; byte Cnt=0;
				if (DPMap[1,0]) { Val+=Data[1,0]; Cnt++; } //6
				if (DPMap[0,1]) { Val+=Data[0,1]; Cnt++; } //8
				if (DPMap[1,1]) { Val+=Data[1,1]; Cnt++; } //9
				if (Cnt!=0) { Data[0,0]=(ushort)(Val/Cnt); }
        	}
        	//oben rechts
        	if (!DPMap[205,0]) {
        		long Val=0; byte Cnt=0;
				if (DPMap[204,0]) { Val+=Data[204,0]; Cnt++; } //4
				if (DPMap[204,1]) { Val+=Data[204,1]; Cnt++; } //7
				if (DPMap[205,1]) { Val+=Data[205,1]; Cnt++; } //8
				if (Cnt!=0) { Data[205,0]=(ushort)(Val/Cnt); }
        	}
        	//unten links
        	if (!DPMap[0,155]) {
        		long Val=0; byte Cnt=0;
				if (DPMap[0,154]) { Val+=Data[0,154]; Cnt++; } //2
				if (DPMap[1,154]) { Val+=Data[1,154]; Cnt++; } //3
				if (DPMap[1,155]) { Val+=Data[1,155]; Cnt++; } //6
				if (Cnt!=0) { Data[0,155]=(ushort)(Val/Cnt); }
        	}
        	//unten rechts
        	if (!DPMap[205,155]) {
        		long Val=0; byte Cnt=0;
				if (DPMap[204,154]) { Val+=Data[204,154]; Cnt++; } //1
				if (DPMap[205,154]) { Val+=Data[205,154]; Cnt++; } //2
				if (DPMap[204,155]) { Val+=Data[204,155]; Cnt++; } //4
				if (Cnt!=0) { Data[205,155]=(ushort)(Val/Cnt); }
        	}
        }
        public void Frame_CalcMinMax(bool[,] DPMap)
        {
        	long AVR=0; ushort cnt=0;
        	MinValue=0xffff;
        	MaxValue=0;
        	for (int y = 0; y < 156; ++y) {
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
        			AVR+=Data[x,y]; cnt++;
					if (Data[x,y] < MinValue) { MinValue = Data[x,y]; }
			        if (Data[x,y] > MaxValue) { MaxValue = Data[x,y]; }
				}
			}
        	if (cnt>0) {
				AvrValue=(ushort)(AVR/(long)cnt);
			}
        }
        public void Frame_Offsetcal(bool[,] DPMap,int[,] OMap)
        {
        	for (int y = 0; y < 156; ++y) {
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
        			int val =Data[x,y]-OMap[x,y];
        			if (val<0) { val=0; } if (val>0xffff) { val=0xffff; }
        			Data[x,y]=(ushort)val;
				}
			}
        }
        public void Frame_OffsetLinecal(bool[,] DPMap,int[,] OMap)
        {
        	for (int y = 0; y < 156; ++y) {
        		int linC=(int)(LineCal[y]-500);
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
        			int val = Data[x,y]+OMap[x,y]+linC;
        			if (val<0) { val=0; } if (val>0xffff) { val=0xffff; }
        			Data[x,y]=(ushort)val;
				}
			}
        }
        public void Frame_Shuttercal(bool[,] DPMap,ushort[,] Shutter,int avr,ushort typ)
        {
        	float fakt = ((float)typ/100f);
        	fakt=1f;
        	for (int y = 0; y < 156; ++y) {
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
        			int val = (int)Data[x,y];
        			val = (int)((Data[x,y])-((float)(Shutter[x,y]-avr)*fakt))+AvrValue;
//        			switch (typ) {
//        				case 0: break;
//        				case 1: val = (int)((Data[x,y])+(Shutter[x,y]))-AvrValue; break;
//        				case 2: val = (int)((Data[x,y])+(Shutter[x,y])+(Shutter[x,y]))+AvrValue;break;
//        				case 3: val = (int)((Data[x,y])-(Shutter[x,y])-(Shutter[x,y])-(Shutter[x,y]))+AvrValue;break;
//        				case 4: val = (int)((Data[x,y])-(0-Shutter[x,y]))+AvrValue;break;
//        				case 5: val = (int)((Data[x,y]));break;
//        			}
        			
        			if (val<0) { val=0; } if (val>0xffff) { val=0xffff; }
        			Data[x,y]=(ushort)val;
				}
			}
        }
        public int[,] Cal_GetOffsets(bool[,] DPMap,bool[,] DPMapBase,int avr)
        {
        	int[,] output = new int[206,156];
        	for (int y = 0; y < 156; ++y) {
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]||!DPMapBase[x,y]) { continue; } //invalid ignorieren
        			output[x,y] = Data[x,y]-avr;
				}
			}
        	return output;
        }
        public bool[,] Cal_GetDefPixel(bool[,] DPMapBase,ushort avr)
        { //true = valid und false = defekt
        	bool[,] output = new bool[206,156];
        	if (DPMapBase==null) {
        		//first run Frame id4
	        	for (int y = 0; y < 156; ++y) {
					for (int x = 0; x < 206; ++x) {
	        			if ((Data[x,y] < 2000) || (Data[x,y] > 12000)) {
	        				output[x,y]=false;
	        			} else {
	        				output[x,y]=!is_pattern_pixel(x,y);
	        			}
					}
				}
        	} else { //ab jetzt bei shutter cal
	        	for (int y = 0; y < 156; ++y) {
					for (int x = 0; x < 206; ++x) {
	        			//output[x,y]=true;
	        			int diff = Data[x,y]-AvrValue;
	        			if (diff<0) { diff=0-diff; }
	        			if (diff<2000) {//&&DPMapBase[x,y]
//	        				if (DPMapBase[x,y]) {  ; }
	        				output[x,y]=true;
	        			}
					}
				}
        	}
        	return output;
        }
        bool is_pattern_pixel(int x, int y)
		{
		    int pattern_start = (10 - y * 4) % 15;
		    if (pattern_start < 0) { pattern_start = 15 + pattern_start; }
		    return (x >= pattern_start && ((x - pattern_start) % 15) == 0);
		}
        public void Frame_Gaincal(bool[,] DPMap,double[,] GMap,bool invert,ushort faktor)
        {
        	float fakt = ((float)faktor/100f);
        	if (invert) {
        		for (int y = 0; y < 156; ++y) {
					for (int x = 0; x < 206; ++x) {
	        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
	        			int val = (int)((double)Data[x,y]/(GMap[x,y]));
	        			val=(int)(val*fakt);
	        			if (val<0) { val=0; } if (val>0xffff) { val=0xffff; }
	        			Data[x,y]=(ushort)val;
					}
				}
        	} else {
	        	for (int y = 0; y < 156; ++y) {
					for (int x = 0; x < 206; ++x) {
	        			if (!DPMap[x,y]) { continue; } //invalid ignorieren
	        			int val = (int)((double)Data[x,y]*GMap[x,y]);
	        			val=(int)(val*fakt);
	        			if (val<0) { val=0; } if (val>0xffff) { val=0xffff; }
	        			Data[x,y]=(ushort)val;
					}
				}
        	}
        	
        }
        public double[,] Cal_GetGains(bool[,] DPMap)
        {
        	double[,] output = new double[206,156];
        	for (int y = 0; y < 156; ++y) {
				for (int x = 0; x < 206; ++x) {
        			if (!DPMap[x,y]) { output[x,y]=1f; continue; } //invalid ignorieren
        			float source = Data[x,y];
        			if (source>1000&&source<12000) {
        				double val = (double)(AvrValue)/(double)(source);
        				if (val<-10||val>10) { val=1f; }
        				if (val==0) { val=1f; }
        				output[x,y]=(double)val;
        			} else {
        				output[x,y]=1f;
        			}
				}
			}
        	return output;
        }
        
        #region Old_Test
        
        
        public void old_applyGainCalibration(double[] calibration)
        {
        	for (int i=0;i< RawDataU16.Length;i++ ) {
        		RawDataU16[i]=(ushort)(RawDataU16[i]*calibration[i]);
        	}
        }
        
        
        public ushort[] ProcessFrameU16(ThermalFrame calibrationFrame, ThermalFrame frameID4)
        {
        	if (calibrationFrame==null||frameID4==null) { return null; }
        	ushort[] output = new ushort[(Width * Height)+2];
			ushort min = 0xFFFF;
            ushort max = 0x0000;
                        
            for(int i=0;i<output.Length-2;i++) {
                long v = RawDataU16[i];
                long c = calibrationFrame.RawDataU16[i];
                long r = frameID4.RawDataU16[i];
//                long div;

                // Only accept values if both the image data and the calibration data are within the accepted range
                // [2000-16383]
//                if ((v < 2000) || (v > 16383) || (c < 2000) || (c > 16383)) {
//                    // Dead pixel, clamp it to zero.
//                    v = 0; continue;
//                } 
//                else {
//                    // Adjust divider for frameID4 based in the sensor data, instead of a constant
//                    // Not sure what value to scale the divider based on the sensor input (v)
//                    // in order to get rid off the horizontal bands at all ranges.
//                    // Also it needs more adjustements because is brighter in the upper left
//                    // div = 11;
//                    
//                }
//                div = v / 2048 + 8;
//                    r /= div;
                    v += 0x4000;
                    v -= c;
                    v += r;
                output[i+2] = (UInt16)v;
           		// 14K to 32K is valid for min/max
           		if (v < 14336) { continue; }
                if (v > 32767) { continue; }
                if (v < min) { min = (UInt16)v; }
                if (v > max) { max = (UInt16)v; }
            }
            output[0] = max;
            output[1] = min;
            //return new CalibratedThermalFrame(output);
            return output;
        }
        public ushort[,] ProcessFrame_S0(ThermalFrame calibrationFrame, ThermalFrame frameID4, ushort[] Parameter)
        { //only the base cal
        	if (calibrationFrame==null||frameID4==null) { return null; }
        	ushort[,] output = new ushort[Width+1,Height+1];
            int x=0,y=0;
            long LastValid = 0;
            for(int i=0;i<RawDataU16.Length;i++) {
                long v = RawDataU16[i];
                long c = calibrationFrame.RawDataU16[i];
                long r = frameID4.RawDataU16[i];
                if ((v < 2000) || (v > 16383) || (c < 2000) || (c > 16383)) {
                    // Dead pixel, clamp it to zero.
                    v=LastValid;
                } else {
	                v += 0x4000;
                    v -= c;
//                    v += r;
                    if (LastValid!=0) {
		                long diff = v-LastValid; if (diff<0) { diff=0-diff; }
		                if (diff>Parameter[0]) {
		                	v=LastValid;
		                	if (Parameter[1]==5) {
		                    	v=12000;
		                    }	
		                }
                    }
                 	LastValid=v;
                }
                switch (Parameter[1]) {
                	case 1: v = calibrationFrame.RawDataU16[i]; break;
                	case 2: v = RawDataU16[i]; break;
                	case 3: v = RawDataU16[i];
	                	float cal = (float)frameID4.RawDataU16[i];
	                	if (cal!=0) {
	                		cal=(float)((frameID4.AvrValue-frameID4.MinValue)/(frameID4.RawDataU16[i]-frameID4.MinValue+0.0001f));
	                		int offset = (calibrationFrame.AvrValue-calibrationFrame.RawDataU16[i]);
	                		if (offset<0) { offset=0; }
	                		long Caled = (long)((float)v*cal);
	                		if (Caled<65000) {
	                			v=(ushort)Caled;
	                		} else {
	                			v=calibrationFrame.AvrValue;
	                		}
	                		v+=offset;
	                	}
                			break;
                }
                if (x<Width) {
                	output[x,y] = (UInt16)v;
                }
                
//                y++;
//                if (y==Height) {
//                	x++;
//                	y=0;
//                	if (x==Width+2) {
//                		break;
//                	}
//                }
                x++;
                if (x==Width+2) {
                	y++;
                	x=0;
                	if (y==Height) {
                		break;
                	}
                }
            }
            
            return output;
        }
        public ushort[,] ProcessFrame_S1(ThermalFrame calibrationFrame, ThermalFrame frameID4, short[,] offsetmap)
        { //base cal + offset
        	if (calibrationFrame==null||frameID4==null) { return null; }
        	ushort[,] output = new ushort[Width+1,Height+1];
            int x=0,y=0;
            for(int i=0;i<RawDataU16.Length;i++) {
                long v = RawDataU16[i];
                long c = calibrationFrame.RawDataU16[i];
                long r = frameID4.RawDataU16[i];
                    v += 0x4000;
                    v -= c;
                    v += r;
                    v -= offsetmap[x,y];
                output[x,y] = (UInt16)v;
                x++;
                if (x==Width) {
                	y++;
                	x=0;
                	if (y==Height) {
                		break;
                	}
                }
            }
            return output;
        }
        
        
		bool is_pattern_pixel(int pos)
		{
			return is_pattern_pixel(pos % 206, pos / 206);
		}
        #endregion
        
    }
    
    public class SeekThermalClass
    {
    	
    	WinUSBDevice device; //descr.idVendor == 0x289D && descr.idProduct == 0x0010
    	public readonly int Width=206, Height=156;
    	public ThermalFrame FrameID4;
    	public ThermalFrame FrameID1;
    	public ThermalFrame FrameID6;
    	
    	public ThermalFrame FrameID8;
    	public volatile bool Streaming = false;
    	public byte LastFrameStatusByte=0;
    	
    	
    	public volatile Bitmap LastProcessedImage;
    	public volatile byte[] lastSeekRecivedBytes;
    	public volatile UInt16[,] lastProcessedFrame;
    	public double[,] LastCal_Gain;
    	public int[,] LastCal_Offset=new int[206,156];
    	public bool[,] DeathPixMapBase=new bool[206,156];
    	public bool[,] DeathPixMap=new bool[206,156];
    	byte[] GetFrame_SetupData=new byte[] {0xc0, 0x7e, 0, 0};
    	public ushort[] Parameter = new ushort[] {0,100,30000,30000};
    	
        public static IEnumerable<WinUSBEnumeratedDevice> Enumerate()
        {
            foreach (WinUSBEnumeratedDevice dev in WinUSBDevice.EnumerateAllDevices()) {
                // Seek Thermal "iAP Interface" device - Use Zadig to install winusb driver on it.
                if (dev.VendorID == 0x289D && dev.ProductID == 0x0010 && dev.UsbInterface == 0) {
                    yield return dev;
                }
            }
            yield return null;
        }
        public SeekThermalClass()
        {
        	WinUSBEnumeratedDevice dev = SeekThermalClass.Enumerate().First();
            device = new WinUSBDevice(dev);
            // device setup sequence
//            try {
//                device.ControlTransferOut(0x41, 0x54, 0, 0, new byte[] { 0x01 });
//            } catch {
//                // Try deinit device and repeat.
//                Deinit();
//                device.ControlTransferOut(0x41, 0x54, 0, 0, new byte[] { 0x01 });
//            }

            device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x01 });
            device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x00, 0x00 });
            byte[] data1 = device.ControlTransferIn(0xC1, 0x4e, 0, 0, 4);
            byte[] data2 = device.ControlTransferIn(0xC1, 0x36, 0, 0, 12);

            // Analysis of 0x56 payload: 
            // First byte seems to be half the size of the output data.
            // It seems like this command may be retriving some sensor data?
            device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x06, 0x00, 0x08, 0x00, 0x00, 0x00 });
            byte[] data3 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x0C);
            device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x01, 0x00, 0x00, 0x06, 0x00, 0x00 });
            byte[] data4 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x02);
			device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x01, 0x00, 0x01, 0x06, 0x00, 0x00 });
			byte[] data5 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x02);
			device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x20, 0x00, 0x30, 0x00, 0x00, 0x00 });
			byte[] data6 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x40);
			device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x20, 0x00, 0x50, 0x00, 0x00, 0x00 });
			byte[] data7 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x40);
			device.ControlTransferOut(0x41, 0x56, 0, 0, new byte[] { 0x0C, 0x00, 0x70, 0x00, 0x00, 0x00 });
            byte[] data8 = device.ControlTransferIn(0xC1, 0x58, 0, 0, 0x18);
			device.ControlTransferOut(0x41, 0x3E, 0, 0, new byte[] { 0x08, 0x00 });
            byte[] data9 = device.ControlTransferIn(0xC1, 0x3D, 0, 0, 2);
			device.ControlTransferOut(0x41, 0x3c, 0, 0, new byte[] { 0x01, 0x00 });
            byte[] data10 = device.ControlTransferIn(0xC1, 0x3D, 0, 0, 2);
            
        }
        public void Deinit()
        {
            device.ControlTransferOut(0x41, 0x3C, 0, 0, new byte[] { 0x00, 0x00 });
            device.ControlTransferOut(0x41, 0x3C, 0, 0, new byte[] { 0x00, 0x00 });
            device.ControlTransferOut(0x41, 0x3C, 0, 0, new byte[] { 0x00, 0x00 });
            device.Dispose();
        }
        
        public byte[] GetFrame_asBytes()
        {
            // Request frame (vendor interface request 0x53; data "C0 7e 00 00" which is half the size of the return data)
            device.ControlTransferOut(0x41, 83, 0, 0, GetFrame_SetupData);

            // Read data from IN 1 pipe
            lastSeekRecivedBytes = device.ReadExactPipe(0x81, 0x7ec0 * 2);
            return lastSeekRecivedBytes;
        }
        public void Write(byte req)
        {
            // Request frame (vendor interface request 0x53; data "C0 7e 00 00" which is half the size of the return data)
            device.ControlTransferOut(0x41, req, 1, 0, new byte[]{1,1,1});
        }
        
        #region Processed_byte
        /// <summary>
        /// Returns RAW Framedata as ThermalFrame class.
        /// </summary>
        public ThermalFrame GetFrame_asClass()
        {
            return new ThermalFrame(GetFrame_asBytes());
        }
        
        public void Start_ProcByte_Stream()
		{
        	if (Streaming) { return; }
        	Streaming = true;
        	Thread T = new Thread(ProcByte_Stream_Funktion);
			T.Start();
		}
        void ProcByte_Stream_Funktion()
		{
        	if (FrameID1==null) {
        		Grab_VisualFrame(); //startup with getting 1 full processed frame
        	}
        	if (FrameID4==null) {
        		Streaming=false;
        		return;
        	}
		    while (Streaming) {
        		lastProcessedFrame = Grab_VisualFrame().Data;
        		if (lastProcessedFrame==null) {
        			Streaming = false;
        			break;
        		} else {
        			OnEvent();
        		}
        	}
		}
        #endregion
        
        public ThermalFrame Grab_VisualFrame()
        {
        	ThermalFrame TF=null;
        	
        	int trys = 30;
        	while (trys>0) {
                // Get frame
                trys--;
                TF = GetFrame_asClass();
                if (TF.Data==null) {
                	//System.Windows.Forms.MessageBox.Show("No Framedata Recived.\r\n" + "Reconnect USB and try again.");
                	return new ThermalFrame(new byte[206*156]); 
                }
                LastFrameStatusByte=TF.StatusByte;
                // Keep the first 6 frames, or anytime those frame IDs are encountered.
                // They might be usefull for image processing.
                //The first frames have the following IDs:
				//frame 1: 4		frame 5: 10
				//frame 2: 9		frame 6: 5
				//frame 3: 8		frame 7: 1
				//frame 4: 7		frame 8: 3
				//So, after the initialization sequence, you'll get something like this:
				//6, 1, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 3 etc.
				if (TF.StatusByte==4) {
					//Gain Cal
					FrameID4 = TF;
					DeathPixMapBase=TF.Cal_GetDefPixel(null,FrameID4.AvrValue);
					//DeathPixMap=TF.Cal_GetDefPixel(null,FrameID4.AvrValue);
					TF.Frame_RemoveDeathPixel(DeathPixMapBase);
					TF.Frame_CalcMinMax(DeathPixMapBase);
					LastCal_Gain=TF.Cal_GetGains(DeathPixMapBase);
				} else if (TF.StatusByte==8) {
					//TF.Frame_Gaincal(DeathPixMapBase,LastCal_Gain,false);
//					TF.Frame_RemoveDeathPixel(DeathPixMapBase);
//					TF.Frame_CalcMinMax(DeathPixMapBase);
//					FrameID8=TF;
					FrameID6=TF;
					break;
				} else if (TF.StatusByte==1) {
					if (FrameID6==null) {
						continue;
					}
					// Offset calibration (every time the shutter is heard)
					TF.Frame_Gaincal(DeathPixMapBase,LastCal_Gain,false,100);
					//TF.Frame_Offsetcal(DeathPixMapBase,LastCal_Offset);
					//
					//TF.Frame_Shuttercal(DeathPixMapBase,FrameID6.Data,FrameID6.AvrValue,100);
					//TF.Frame_RemoveDeathPixel(DeathPixMapBase);
					//TF.Frame_RemoveDeathPixel(DeathPixMap);
					TF.Frame_CalcMinMax(DeathPixMapBase);
					//FrameID1 = TF;
					TF.Frame_RemoveDeathPixel(DeathPixMapBase);
					//DeathPixMap=TF.Cal_GetDefPixel(DeathPixMapBase,FrameID1.AvrValue);
					
//					LastCal_Offset = TF.Cal_GetOffsets(DeathPixMapBase,FrameID4.AvrValue);
//					DeathPixMap=TF.Cal_GetDefPixel(DeathPixMapBase,FrameID4.AvrValue);
					FrameID1 = TF;
					//TF.Frame_RemoveDeathPixel(DeathPixMap);
					//LastCal_Offset = TF.Cal_GetOffsets(DeathPixMap,DeathPixMapBase,FrameID1.AvrValue+(Parameter[2]-30000));
					//TF.Frame_Offsetcal(DeathPixMap,LastCal_Offset);
					
					
					break;
				} else if (TF.StatusByte==6) {
					TF=FrameID6;
					//TF.Frame_Gaincal(DeathPixMapBase,LastCal_Gain,false);
					//TF.Frame_Shuttercal(DeathPixMapBase,FrameID8.Data,FrameID4.AvrValue,100);
					//TF.Frame_RemoveDeathPixel(DeathPixMapBase);
					//TF.Frame_RemoveDeathPixel(DeathPixMap);
					//TF.Frame_CalcMinMax(DeathPixMapBase);
					
//					FrameID6=TF;
//					DeathPixMap=TF.Cal_GetDefPixel(DeathPixMapBase,FrameID6.AvrValue);
//					TF.Frame_RemoveDeathPixel(DeathPixMap);
//					FrameID6=TF;
					break;
				} else if (TF.StatusByte==3) {
					//IsUsableFrame
					//TF.Frame_RemoveDeathPixel(DeathPixMap);
					
					if (FrameID6==null) { continue; }
					
					//TF.Frame_CalcMinMax(DeathPixMap);
					TF.Frame_Gaincal(DeathPixMapBase,LastCal_Gain,false,Parameter[1]);
					//TF.Frame_OffsetLinecal(DeathPixMap,LastCal_Offset);
					//
					TF.Frame_Shuttercal(DeathPixMapBase,FrameID1.Data,(Parameter[3]-30000),100);
					//TF.Frame_Offsetcal(DeathPixMapBase,LastCal_Offset);
					//TF.Frame_Shuttercal(DeathPixMapBase,FrameID6.Data,FrameID6.AvrValue,100);
					//
					TF.Frame_CalcMinMax(DeathPixMapBase);
					TF.Frame_RemoveDeathPixel(DeathPixMapBase);
					//TF.Frame_RemoveDeathPixel(DeathPixMap);
					break;
				}
            } //while...
        	if (trys==0) {
        		System.Windows.Forms.MessageBox.Show("Timeout while GetFrame_asClass.");
        		return null;
        	}
        	return TF;
        }
        public ushort[,] DataProc_Median(ushort[,] data)
        {
        	long val=0; int cnt=0;
        	ushort[,] Proc = new ushort[Width,Height];
        	//Frame without edges
        	for (int y = 1; y < 155; y++) {
                for (int x = 1; x < 205; x++) {
                    //1  2  3
                    //4  5  6
                    //7  8  9
                    val=0; cnt=0;
                    val += data[x-1,y-1]; cnt++;  //1
                    val += data[x,y-1]; cnt++;  //2
                    val += data[x+1,y-1]; cnt++;  //3
                    val += data[x-1,y]; cnt++;  //4
                    val += data[x,y]; cnt++;  //5
                    val += data[x+1,y]; cnt++;  //6
                    val += data[x-1,y+1]; cnt++;  //7
                    val += data[x,y+1]; cnt++;  //8
                    val += data[x+1,y+1]; cnt++;  //9
                    Proc[x,y] = (ushort)(val/(long)cnt);
                }
            }
        	//edge W --------------------------------------------
    		for (int y = 1; y < 155; y++) {
                val=0; cnt=0;
                val += data[0,y-1]; cnt++;  //2
                val += data[1,y-1]; cnt++;  //3
                val += data[1,y]; cnt++;  //6
                val += data[0,y+1]; cnt++;  //8
                val += data[1,y+1]; cnt++;  //9
				Proc[0,y] = (ushort)(val/(long)cnt);
            }
        	//edge E --------------------------------------------
    		for (int y = 1; y < 155; y++) {
                val=0; cnt=0;
                val += data[204,y-1]; cnt++;  //1
                val += data[205,y-1]; cnt++;  //2
                val += data[204,y]; cnt++;  //4
                val += data[204,y+1]; cnt++;  //7
                val += data[205,y+1]; cnt++;  //8
                Proc[205,y] = (ushort)(val/(long)cnt); 
            }
        	//edge N --------------------------------------------
    		for (int x = 1; x < 205; x++) {
                val=0; cnt=0;
                val += data[x-1,0]; cnt++;  //4
                val += data[x+1,0]; cnt++;  //6
                val += data[x-1,1]; cnt++;  //7
                val += data[x,1]; cnt++;  //8
                val += data[x+1,1]; cnt++;  //9
				Proc[x,0] = (ushort)(val/(long)cnt); 
            }
        	//edge S --------------------------------------------
    		for (int x = 1; x < 205; x++) {
                val=0; cnt=0;
                val += data[x-1,154]; cnt++;  //1
                val += data[x,154]; cnt++;  //2
                val += data[x+1,154]; cnt++;  //3
                val += data[x-1,155]; cnt++;  //4
                val += data[x+1,155]; cnt++;  //6
				Proc[x,155] = (ushort)(val/(long)cnt); 
            }
        	//übertragen
        	Proc[0,0]=data[0,0];
        	Proc[0,155]=data[0,155];
        	Proc[205,0]=data[205,0];
        	Proc[205,155]=data[205,155];
        	return Proc;
//        	for (int y = 0; y < 156; y++) {
//                for (int x = 0; x < 206; x++) {
//                   data[x,y]=Proc[x,y];
//                }
//            }
        }
        
        #region Processed_Image
        //just transform the bytestream on some default ways to get a easy camera live view
        
        
        
        /// <summary>
        /// Returns a processed Bitmap image (206x156). The Frame is autoscaled Grayscale
        /// and cross-averaged (defect pixel).
        /// </summary>
//        public Bitmap GetFrame_asProcessedImage()
//        {
//        	ThermalFrame TF=Grab_VisualFrame();
//        	if (TF==null) { return null; }
//        	ushort[] Frame = TF.ProcessFrameU16(LastCalframe,FrameID4);
//        	int noiseReduction = 4 * (Frame[0] - Frame[1]) / 256; 
//            ushort[] arrColor = new ushort[5];
//            ushort min = Frame[1];
//        	ushort max = Frame[0];
//            ulong temp,sum;
//            
//			
//        	for (int y = 0; y < 154; y++) {
//                for (int x = 0; x < 205; x++) {
//                    min = 0xffff;
//                    max = 0;
//                    sum = 0;
//                    //   2 
//                    //0  d  1
//                    //   3
//                    arrColor[0] = Frame[2 + x + 206 * (y + 1)];
//                    arrColor[1] = Frame[2 + x + 2 + 206 * (y + 1)];
//                    arrColor[2] = Frame[2 + x + 1 + 206 * (y)];
//                    arrColor[3] = Frame[2 + x + 1 + 206 * (y + 2)];
//                    for (int i = 0; i < 4; i++) {
//                        if (arrColor[i] > max) max = arrColor[i];
//                        if (arrColor[i] < min) min = arrColor[i];
//                        sum += arrColor[i];
//                    }
//
//                    //get average value, but exclude neighbour dead pixels from average:
//                    temp = min;
//                    temp += max;
//                    ulong Pixdata = sum;
//                    Pixdata -= temp;
//                    Pixdata /= 2;
//
//                    arrColor[4] = Frame[2 + x + 1 + 206 * (y + 1)];
//                    if (Math.Abs((ushort)Pixdata - arrColor[4]) > noiseReduction) {
//                    	//if image is too different use average color from near pixels
//                        Frame[2 + x + 1 + 206 * (y + 1)] = (ushort)Pixdata;
//                    }
//                }
//            }
//
//            // Clean up the borders so we don't have strange data
//            // we loose the original value and we use the interpolation of the pixels next to the edge
//            // for all the edges we care about (top bottom and left).
//            ulong average;
//            int c;
//            
//            for (int x = 1; x < 206; x++) {
//                c=x;
//            	//c = y * 208 + x;
//                average = Frame[2+c + 207];
//                average += Frame[2+c + 208];
//                average += Frame[2+c + 209];
//                average /= 3;
//                Frame[2+c] = (ushort)average;
//            }
//           // y = 155;
//            for (int x = 1; x < 205; x++) {
//                c = 155 * 206 + x;
//                average = Frame[2+c - 207];
//                average += Frame[2+c - 208];
//                average += Frame[2+c - 209];
//                average /= 3;
//                Frame[c] = (ushort)average;
//            }
//            for (int y = 1; y < 155; y++) {
//                c = y * 206;
//                average = Frame[2+c - 207];
//                average += Frame[2+c + 1];
//                average += Frame[2+c + 209];
//                average /= 3;
//                Frame[2+c] = (ushort)average;
//            }
//            
//            // Recompute the min max after the image is all fixed
//            ushort maxValue=0, minValue=0xFFFF;
//            for (int y = 0; y < 156; y++) {
//                for (int x = 0; x < 206; x++) {
//                    ushort val = Frame[2+ y * 206 + x];
//                    if (val < minValue) minValue = val;
//                    if (val > maxValue) maxValue = val;
//                }
//            }
//            
//            //Draw Picture
//			PixelData Col = new PixelData();
//        	UnsafeBitmap ubmp = new UnsafeBitmap(206, 156);
//        	ubmp.LockBitmap();
//        	for (int y = 0; y < 156; y++) {
//                for (int x = 0; x < 206; x++) {
//					ushort val = Frame[2+ y * 206 + x];
//					
//					ushort offset = (ushort)(maxValue-minValue);
//                    if (offset == 0) { offset = 1; } // Avoid divide by 0
//                    
//        			int Pixdata = (int)(((float)(val-minValue)*256f/(float)offset));
//        			
//        			if (Pixdata>255) { Pixdata=255; }
//        			if (Pixdata<0) { Pixdata=0; }
//        			Col.red=(byte)Pixdata;
//        			Col.green=(byte)Pixdata;
//        			Col.blue=(byte)Pixdata;
//        			ubmp.SetPixel(x,y,Col);
//                }
//            }
//        	ubmp.UnlockBitmap();
////        	lastProcessedFrame=Frame;
//        	return ubmp.Bitmap;
//        }
//        
//        public void Start_ProcImage_Stream()
//		{
//        	if (Streaming) { return; }
//        	Streaming = true;
//        	Thread T = new Thread(ProcImage_Stream_Funktion);
//			T.Start();
//		}
//        void ProcImage_Stream_Funktion()
//		{
//		     while (Streaming) {
//        		Bitmap bmp = GetFrame_asProcessedImage();
//        		if (bmp==null) {
//        			Streaming = false;
//        			break;
//        		} else {
//        			LastProcessedImage = bmp;
//        			OnEvent();
//        		}
//        	}
//		}
		#endregion
		public delegate void EventDelegate();
		public event EventDelegate SeekStreamEvent;// Das Event-Objekt ist vom Typ dieses Delegaten
		public void OnEvent()
		{
			if(SeekStreamEvent != null) { SeekStreamEvent(); }
		}
		
    }
    
    #region NativeMethods
    internal class EnumeratedDevice
    {
        public string DevicePath { get; set; }
        public Guid InterfaceGuid { get; set; }
        public string DeviceDescription { get; set; }
        public string Manufacturer { get; set; }
        public string FriendlyName { get; set; }
    }
    internal class NativeMethods
    {
        private struct SP_DEVINFO_DATA
        {
            public UInt32 cbSize;
            public Guid classGuid;
            public UInt32 devInst;
            public IntPtr reserved;
        }
        private struct SP_DEVICE_INTERFACE_DATA
        {
            public UInt32 cbSize;
            public Guid interfaceClassGuid;
            public UInt32 flags;
            public IntPtr reserved;
        }
        private struct DEVPROPKEY
        {
            public Guid fmtId;
            public UInt32 pId;


            //DEFINE_DEVPROPKEY(DEVPKEY_Device_DeviceDesc,             0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 2);     // DEVPROP_TYPE_STRING
            //DEFINE_DEVPROPKEY(DEVPKEY_Device_Manufacturer,           0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 13);    // DEVPROP_TYPE_STRING
            //DEFINE_DEVPROPKEY(DEVPKEY_Device_FriendlyName,           0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 14);    // DEVPROP_TYPE_STRING

            public static DEVPROPKEY Device_DeviceDesc { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 2 }; } }
            public static DEVPROPKEY Device_Manufacturer { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 13 }; } }
            public static DEVPROPKEY Device_FriendlyName { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 14 }; } }
        }
		public struct WINUSB_SETUP_PACKET
        {
            public byte RequestType;
            public byte Request;
            public UInt16 Value;
            public UInt16 Index;
            public UInt16 Length;
        }
        
        #region const...
        private const UInt32 DIGCF_PRESENT = 2;
        private const UInt32 DIGCF_ALLCLASSES = 4;
        private const UInt32 DIGCF_DEVICEINTERFACE = 0x10;

        private static IntPtr INVALID_HANDLE_VALUE = new IntPtr(-1);

        private const int ERROR_NOT_FOUND = 1168;
        private const int ERROR_FILE_NOT_FOUND = 2;
        private const int ERROR_NO_MORE_ITEMS = 259;
        private const int ERROR_INSUFFICIENT_BUFFER = 122;
        private const int ERROR_MORE_DATA = 234;

        public const int ERROR_SEM_TIMEOUT = 121;
        public const int ERROR_IO_PENDING = 997;

        private const int DICS_FLAG_GLOBAL = 1;
        private const int DICS_FLAG_CONFIGSPECIFIC = 2;

        private const int DIREG_DEV = 1;
        private const int DIREG_DRV = 2;

        private const int KEY_READ = 0x20019; // Registry SAM value.

        private const int RRF_RT_REG_SZ = 2;
        private const int RRF_RT_REG_MULTI_SZ = 0x20;
        
        public const uint FILE_ATTRIBUTE_NORMAL = 0x80;
        public const uint FILE_FLAG_OVERLAPPED = 0x40000000;
        public const uint GENERIC_READ = 0x80000000;
        public const uint GENERIC_WRITE = 0x40000000;
        public const uint CREATE_NEW = 1;
        public const uint CREATE_ALWAYS = 2;
        public const uint OPEN_EXISTING = 3;
        public const uint FILE_SHARE_READ = 1;
        public const uint FILE_SHARE_WRITE = 2;
		#endregion

        /// <summary>
        /// Retrieve device paths that can be opened from a specific device interface guid.
        /// todo: Make this friendlier & query some more data about the devices being returned.
        /// </summary>
        /// <param name="deviceInterface">Guid uniquely identifying the interface to search for</param>
        /// <returns>List of device paths that can be opened with CreateFile</returns>
        public static EnumeratedDevice[] EnumerateDevicesByInterface(Guid deviceInterface)
        {
            // Horribe horrible things have to be done with SetupDI here. These travesties must never leave this class.
            List<EnumeratedDevice> outputPaths = new List<EnumeratedDevice>();

            IntPtr devInfo = SetupDiGetClassDevs(ref deviceInterface, null, IntPtr.Zero, DIGCF_DEVICEINTERFACE | DIGCF_PRESENT);
            if(devInfo == INVALID_HANDLE_VALUE)
            {
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try
            {
                uint deviceIndex = 0;
                SP_DEVICE_INTERFACE_DATA interfaceData = new SP_DEVICE_INTERFACE_DATA();

                bool success = true;
                for (deviceIndex = 0; ; deviceIndex++)
                {
                    interfaceData.cbSize = (uint)Marshal.SizeOf(interfaceData);
                    success = SetupDiEnumDeviceInterfaces(devInfo, IntPtr.Zero, ref deviceInterface, deviceIndex, ref interfaceData);
                    if (!success)
                    {
                        if (Marshal.GetLastWin32Error() != ERROR_NO_MORE_ITEMS)
                        {
                            throw new Exception("SetupDiEnumDeviceInterfaces failed " + (new Win32Exception()).ToString());
                        }
                        // We have reached the end of the list of devices.
                        break;
                    }

                    // This is a valid interface, retrieve its path
                    EnumeratedDevice dev = new EnumeratedDevice() { DevicePath = RetrieveDeviceInstancePath(devInfo, interfaceData), InterfaceGuid = deviceInterface };


                    // Todo: debug. Not working correctly.
                    /*
                    dev.DeviceDescription = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_DeviceDesc);
                    dev.Manufacturer = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_Manufacturer);
                    dev.FriendlyName = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_FriendlyName);
                    */

                    outputPaths.Add(dev);
                }
            }
            finally
            {
                SetupDiDestroyDeviceInfoList(devInfo);
            }

            return outputPaths.ToArray();
        }
        public static EnumeratedDevice[] EnumerateAllWinUsbDevices()
        {
            List<EnumeratedDevice> outputDevices = new List<EnumeratedDevice>();
            string[] guids = EnumerateAllWinUsbGuids();
            foreach (string guid in guids)
            {
                try
                {
                    Guid g = new Guid(guid);
                    outputDevices.AddRange(EnumerateDevicesByInterface(g));
                }
                catch
                {
                    // Ignore failing guid conversions.
                }
            }
            return outputDevices.ToArray();
        }
        public static string[] EnumerateAllWinUsbGuids()
        {
            // Horribe horrible things have to be done with SetupDI here. These travesties must never leave this class.
            List<string> outputGuids = new List<string>();

            IntPtr devInfo = SetupDiGetClassDevs(IntPtr.Zero, null, IntPtr.Zero, DIGCF_ALLCLASSES | DIGCF_PRESENT);
            if (devInfo == INVALID_HANDLE_VALUE) {
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try {
                uint deviceIndex = 0;
                SP_DEVINFO_DATA devInfoData = new SP_DEVINFO_DATA();

                bool success = true;
                for (deviceIndex = 0; ; deviceIndex++) {
                    devInfoData.cbSize = (uint)Marshal.SizeOf(devInfoData);
                    success = SetupDiEnumDeviceInfo(devInfo, deviceIndex, ref devInfoData);
                    if (!success) {
                        if (Marshal.GetLastWin32Error() != ERROR_NO_MORE_ITEMS) {
                            throw new Exception("SetupDiEnumDeviceInfo failed " + (new Win32Exception()).ToString());
                        }
                        // We have reached the end of the list of devices.
                        break;
                    }

                    // Enumerate the WinUSB Interface Guids (if present) by looking at the registry.
                    //DebugEnumRegistryValues(devInfo, devInfoData);
                    string guid = RetrieveDeviceProperty(devInfo, devInfoData, "DeviceInterfaceGUIDs");
                    if(guid==null||guid.EndsWith("!")) {
                        guid = RetrieveDeviceProperty(devInfo, devInfoData, "DeviceInterfaceGUID");
                    }

                    if (guid!=null&&!guid.EndsWith("!")) {
                        outputGuids.Add(guid);
                    }
                }
            }
            finally
            {
                SetupDiDestroyDeviceInfoList(devInfo);
            }

            return outputGuids.Distinct().ToArray();
        }

        static void DebugEnumRegistryValues(IntPtr devInfo, SP_DEVINFO_DATA devInfoData)
        {
            System.Console.WriteLine("DebugEnumRegistryValues");
            IntPtr hKey = SetupDiOpenDevRegKey(devInfo, ref devInfoData, DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_READ);
            if (hKey == INVALID_HANDLE_VALUE)
            {
                System.Console.WriteLine("Failed");
                return;
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try
            {

                IntPtr memValue = Marshal.AllocHGlobal(16384);
                try
                {
                    IntPtr memData = Marshal.AllocHGlobal(65536);
                    try
                    {

                        for (int i = 0; ; i++)
                        {
                            UInt32 outValueLen = 16384;
                            UInt32 outDataLen = 65536;
                            UInt32 outType;
                            long output = RegEnumValue(hKey, (uint)i, memValue, ref outValueLen, IntPtr.Zero, out outType, memData, ref outDataLen);
                            if((int)output == ERROR_NO_MORE_ITEMS)
                            {
                                break;
                            }
                            if (output != 0)
                            {
                                throw new Exception("RegEnumValue failed " + (new Win32Exception((int)output)).ToString());
                            }

                            string value = ReadAsciiString(memValue, (int)outValueLen,0);
                            string data = ReadAsciiString(memData, (int)outDataLen,0);

                            Console.WriteLine("Enum: '{0}' -  {2} '{1}'", value, data, outType);

                        }

                    }
                    finally
                    {
                        Marshal.FreeHGlobal(memData);
                    }
                }
                finally
                {
                    Marshal.FreeHGlobal(memValue);
                }

            }
            finally
            {
                RegCloseKey(hKey);
            }
        }
        static string RetrieveDeviceProperty(IntPtr devInfo, SP_DEVINFO_DATA devInfoData, string deviceProperty)
        {
            IntPtr hKey = SetupDiOpenDevRegKey(devInfo, ref devInfoData, DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_READ);
            if (hKey == INVALID_HANDLE_VALUE) {
                return null; // Ignore failures, probably the key doesn't exist.
            }

            try {
                UInt32 outType;
                UInt32 outLength = 0;
                //static extern int RegQueryValueEx( UIntPtr hKey, string lpValueName, int lpReserved, out uint lpType, System.Text.StringBuilder lpData, ref uint lpcbData);
                //long output = RegGetValueA(hKey,null, deviceProperty, RRF_RT_REG_SZ | RRF_RT_REG_MULTI_SZ, out outType, IntPtr.Zero, ref outLength);
                long output = RegQueryValueEx(hKey,deviceProperty,0,out outType, IntPtr.Zero,out outLength);
                if(output == ERROR_FILE_NOT_FOUND) { //output==2
                    return null; // Key not present, don't continue.
                }
                
                if (output != 0) {
                	return "RegGetValue failed (determining length): "+output.ToString()+" !";
                }
                //output==0
                //deviceProperty==DeviceInterfaceGUIDs
                //outType==7
                //outLength=42
                IntPtr mem = Marshal.AllocHGlobal((int)outLength);
                try {

                    UInt32 actualLength = outLength;
                    //output = RegGetValueA(hKey, null, deviceProperty, RRF_RT_REG_SZ | RRF_RT_REG_MULTI_SZ, out outType, mem, ref actualLength);
	                output = RegQueryValueEx(hKey,deviceProperty,0,out outType, mem,out actualLength);
                    //erster run: outType=7 mem=6786192,12011880 actualLength=40
	                //zweiter run: outType=7 mem=7026584,12275432 actualLength=40
                    if (output != 0) {
                    	return "RegGetValue failed (retrieving data): "+output.ToString()+" !";
                    }

                    // Convert TCHAR string into chars.
                    if (actualLength > outLength) {
                    	return "Consistency issue: Actual length should not be larger than buffer size. !";
                    }

                    return ReadAsciiString(mem, (int)((actualLength)),0);
                } finally {
                    Marshal.FreeHGlobal(mem);
                }
            } catch (Exception err){
            	System.Windows.Forms.MessageBox.Show(err.Message,"Error in RetrieveDeviceProperty");
            } finally {
                RegCloseKey(hKey);
            }
            return null;
        }
        static string RetrieveDeviceInstancePath(IntPtr devInfo, SP_DEVICE_INTERFACE_DATA interfaceData)
        {
            // This is a valid interface, retrieve its path
            UInt32 requiredLength = 0;

            if (!SetupDiGetDeviceInterfaceDetail(devInfo, ref interfaceData, IntPtr.Zero, 0, ref requiredLength, IntPtr.Zero)) {
                int err = Marshal.GetLastWin32Error();

                if (err != ERROR_INSUFFICIENT_BUFFER) {
                    throw new Exception("SetupDiGetDeviceInterfaceDetail failed (determining length) " + (new Win32Exception()).ToString());
                }
            }

            UInt32 actualLength = requiredLength;
            Int32 structLen = 6;
            if (IntPtr.Size == 8) structLen = 8; // Compensate for 64bit struct packing.

            if (requiredLength < structLen) {
                throw new Exception("Consistency issue: Required memory size should be larger");
            }

            IntPtr mem = Marshal.AllocHGlobal((int)requiredLength);
            
            try
            {
                Marshal.WriteInt32(mem, structLen); // set fake size in fake structure

                if (!SetupDiGetDeviceInterfaceDetail(devInfo, ref interfaceData, mem, requiredLength, ref actualLength, IntPtr.Zero))
                {
                    throw new Exception("SetupDiGetDeviceInterfaceDetail failed (retrieving data) " + (new Win32Exception()).ToString());
                }

                // Convert TCHAR string into chars.
                if (actualLength > requiredLength)
                {
                    throw new Exception("Consistency issue: Actual length should not be larger than buffer size.");
                }

                return ReadString(mem, (int)((actualLength - 4) / 2), 4);
            }
            finally
            {
                Marshal.FreeHGlobal(mem);
            }
        }
        static string RetrieveDeviceInstancePropertyString(IntPtr devInfo, SP_DEVICE_INTERFACE_DATA interfaceData, DEVPROPKEY property)
        {
            // This is a valid interface, retrieve its path
            UInt32 requiredLength = 0;
            UInt32 propertyType;

            if (!SetupDiGetDeviceInterfaceProperty(devInfo, ref interfaceData, ref property, out propertyType, IntPtr.Zero, 0, out requiredLength, 0))
            {
                int err = Marshal.GetLastWin32Error();
                if (err == ERROR_NOT_FOUND)
                {
                    return null;
                }

                if (err != ERROR_INSUFFICIENT_BUFFER)
                {
                    throw new Exception("SetupDiGetDeviceInterfaceProperty failed (determining length) " + (new Win32Exception()).ToString());
                }

            }

            UInt32 actualLength = requiredLength;


            IntPtr mem = Marshal.AllocHGlobal((int)requiredLength);
            try
            {
                Marshal.WriteInt32(mem, 6); // set fake size in fake structure

                if (!SetupDiGetDeviceInterfaceProperty(devInfo, ref interfaceData, ref property, out propertyType, mem, requiredLength, out actualLength, 0))
                {
                    throw new Exception("SetupDiGetDeviceInterfaceProperty failed (retrieving data) " + (new Win32Exception()).ToString());
                }

                // Convert TCHAR string into chars.
                if (actualLength > requiredLength)
                {
                    throw new Exception("Consistency issue: Actual length should not be larger than buffer size.");
                }

                return ReadString(mem, (int)((actualLength) / 2),0);
            }
            finally
            {
                Marshal.FreeHGlobal(mem);
            }
        }
        static string ReadString(IntPtr source, int length, int offset)
        {//IntPtr source, int length, int offset = 0
            char[] stringChars = new char[length];
            for (int i = 0; i < length; i++)
            {
                stringChars[i] = (char)Marshal.ReadInt16(source, i * 2 + offset);
                if (stringChars[i] == 0) { length = i; break; }
            }
            return new string(stringChars, 0, length);
        }
        static string ReadAsciiString(IntPtr source, int length, int offset)
        {//IntPtr source, int length, int offset = 0
            char[] stringChars = new char[length];
            for (int i = 0; i < length; i++)
            {
                stringChars[i] = (char)Marshal.ReadByte(source, i + offset);
                if (stringChars[i] == 0) { length = i; break; }
            }
            return new string(stringChars, 0, length);
        }
	
        #region DLL_imports
        /*
        HDEVINFO SetupDiGetClassDevs(
          _In_opt_  const GUID *ClassGuid,
          _In_opt_  PCTSTR Enumerator,
          _In_opt_  HWND hwndParent,
          _In_      DWORD Flags
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static IntPtr SetupDiGetClassDevs(ref Guid classGuid, string enumerator, IntPtr hwndParent, UInt32 flags);
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static IntPtr SetupDiGetClassDevs(IntPtr classGuid, string enumerator, IntPtr hwndParent, UInt32 flags);

        /*
         BOOL SetupDiEnumDeviceInterfaces(
          _In_      HDEVINFO DeviceInfoSet,
          _In_opt_  PSP_DEVINFO_DATA DeviceInfoData,
          _In_      const GUID *InterfaceClassGuid,
          _In_      DWORD MemberIndex,
          _Out_     PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, IntPtr optDeviceInfoData, ref Guid interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, IntPtr optDeviceInfoData, IntPtr interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, ref Guid interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, IntPtr interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);


        /*
        BOOL SetupDiEnumDeviceInfo(
          _In_   HDEVINFO DeviceInfoSet,
          _In_   DWORD MemberIndex,
          _Out_  PSP_DEVINFO_DATA DeviceInfoData
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiEnumDeviceInfo(IntPtr deviceInfoSet, UInt32 memberIndex, ref SP_DEVINFO_DATA deviceInfoData);

        /*
        HKEY SetupDiOpenDevRegKey(
          _In_  HDEVINFO DeviceInfoSet,
          _In_  PSP_DEVINFO_DATA DeviceInfoData,
          _In_  DWORD Scope,
          _In_  DWORD HwProfile,
          _In_  DWORD KeyType,
          _In_  REGSAM samDesired
        );
        */
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static IntPtr SetupDiOpenDevRegKey(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, UInt32 scope, UInt32 hwProfile, UInt32 keyType, UInt32 samDesired);

        /*
        BOOL SetupDiGetDeviceInterfaceProperty(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
          _In_       const DEVPROPKEY *PropertyKey,
          _Out_      DEVPROPTYPE *PropertyType,
          _Out_      PBYTE PropertyBuffer,
          _In_       DWORD PropertyBufferSize,
          _Out_opt_  PDWORD RequiredSize,
          _In_       DWORD Flags
        );
        */
        [DllImport("setupapi.dll", SetLastError = true, CharSet=CharSet.Unicode, EntryPoint="SetupDiGetDeviceInterfacePropertyW")]
        private extern static bool SetupDiGetDeviceInterfaceProperty(IntPtr deviceInfoSet, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceDataa, ref DEVPROPKEY propertyKey, out UInt32 propertyType, IntPtr outPropertyData, UInt32 dataBufferLength, out UInt32 requredBufferLength, UInt32 flags);


        /*
        BOOL SetupDiGetDevicePropertyKeys(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVINFO_DATA DeviceInfoData,
          _Out_opt_  DEVPROPKEY *PropertyKeyArray,
          _In_       DWORD PropertyKeyCount,
          _Out_opt_  PDWORD RequiredPropertyKeyCount,
          _In_       DWORD Flags
        );
        */

        /*
        LONG WINAPI RegCloseKey(
        _In_  HKEY hKey
        );
        */
        [DllImport("advapi32.dll", SetLastError = false)]
        private extern static int RegCloseKey(IntPtr hKey);


        /*
        LONG WINAPI RegGetValue(
          _In_         HKEY hkey,
          _In_opt_     LPCTSTR lpSubKey,
          _In_opt_     LPCTSTR lpValue,
          _In_opt_     DWORD dwFlags,
          _Out_opt_    LPDWORD pdwType,
          _Out_opt_    PVOID pvData,
          _Inout_opt_  LPDWORD pcbData
        );
        */
        [DllImport("advapi32.dll", SetLastError = false)]//RegGetValue
        private extern static int RegGetValueA(IntPtr hKey, string lpSubKey, string lpValue, UInt32 flags, out UInt32 outType, IntPtr outData, ref UInt32 dataLength);
//		
        [DllImport("advapi32.dll",EntryPoint="RegQueryValueEx")]
		public static extern int RegQueryValueEx(IntPtr hKey,string lpValueName,int lpReserved,out uint lpType, IntPtr lpData,out uint lpcbData);

        //[DllImport("advapi32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
//		static extern int RegGetValue( IntPtr hKey, string lpValueName, int lpReserved, out uint lpType, System.Text.StringBuilder lpData, ref uint lpcbData);
//        /*
//        LONG WINAPI RegEnumValue(
//          _In_         HKEY hKey,
//          _In_         DWORD dwIndex,
//          _Out_        LPTSTR lpValueName,
//          _Inout_      LPDWORD lpcchValueName,
//          _Reserved_   LPDWORD lpReserved,
//          _Out_opt_    LPDWORD lpType,
//          _Out_opt_    LPBYTE lpData,
//          _Inout_opt_  LPDWORD lpcbData
//        );
//        */
        [DllImport("advapi32.dll", SetLastError = false)]
        private extern static int RegEnumValue(IntPtr hKey, UInt32 index, IntPtr outValue, ref UInt32 valueLen, IntPtr reserved, out UInt32 outType, IntPtr outData, ref UInt32 dataLength);

		
 
 /*
        BOOL SetupDiDestroyDeviceInfoList(
          _In_  HDEVINFO DeviceInfoSet
        );
          */
        [DllImport("setupapi.dll", SetLastError = true)]
        private extern static bool SetupDiDestroyDeviceInfoList(IntPtr deviceInfoSet);


        /* 
        BOOL SetupDiGetDeviceInterfaceDetail(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
          _Out_opt_  PSP_DEVICE_INTERFACE_DETAIL_DATA DeviceInterfaceDetailData,
          _In_       DWORD DeviceInterfaceDetailDataSize,
          _Out_opt_  PDWORD RequiredSize,
          _Out_opt_  PSP_DEVINFO_DATA DeviceInfoData
        );
          */
        [DllImport("setupapi.dll", SetLastError = true, CharSet=CharSet.Unicode)]
        private extern static bool SetupDiGetDeviceInterfaceDetail(IntPtr deviceInfoSet, [In] ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData, 
            IntPtr deviceInterfaceDetailData, UInt32 deviceInterfaceDetailSize, ref UInt32 requiredSize, IntPtr deviceInfoData );


        /* 
        HANDLE WINAPI CreateFile(
          _In_      LPCTSTR lpFileName,
          _In_      DWORD dwDesiredAccess,
          _In_      DWORD dwShareMode,
          _In_opt_  LPSECURITY_ATTRIBUTES lpSecurityAttributes,
          _In_      DWORD dwCreationDisposition,
          _In_      DWORD dwFlagsAndAttributes,
          _In_opt_  HANDLE hTemplateFile
        );
          */
        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        public extern static SafeFileHandle CreateFile(string lpFileName, UInt32 dwDesiredAccess, 
            UInt32 dwShareMode, IntPtr lpSecurityAttributes, UInt32 dwCreationDisposition, UInt32 dwFlagsAndAttributes, IntPtr hTemplateFile);
		/* 
        BOOL __stdcall WinUsb_Initialize(
          _In_   HANDLE DeviceHandle,
          _Out_  PWINUSB_INTERFACE_HANDLE InterfaceHandle
        );
          */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_Initialize(SafeFileHandle deviceHandle, out IntPtr interfaceHandle);

        /* 
        BOOL __stdcall WinUsb_Free(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_Free(IntPtr interfaceHandle);

        /*
         BOOL __stdcall WinUsb_ControlTransfer(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       WINUSB_SETUP_PACKET SetupPacket,
          _Out_      PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ControlTransfer(IntPtr interfaceHandle, WINUSB_SETUP_PACKET setupPacket, byte[] buffer, uint bufferLength, out UInt32 lengthTransferred, IntPtr overlapped);

        /* 
        BOOL __stdcall WinUsb_ReadPipe(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       UCHAR PipeID,
          _Out_      PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ReadPipe(IntPtr interfaceHandle, byte pipeId, IntPtr buffer, uint bufferLength, IntPtr lengthTransferred, IntPtr overlapped);
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ReadPipe(IntPtr interfaceHandle, byte pipeId, [Out] byte[] buffer, uint bufferLength, ref UInt32 lengthTransferred, IntPtr overlapped);

        /* 
        BOOL __stdcall WinUsb_WritePipe(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       UCHAR PipeID,
          _In_       PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_WritePipe(IntPtr interfaceHandle, byte pipeId, [In] byte[] buffer, uint bufferLength, IntPtr lengthTransferred, ref NativeOverlapped overlapped);
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_WritePipe(IntPtr interfaceHandle, byte pipeId, [In] byte[] buffer, uint bufferLength, ref UInt32 lengthTransferred, IntPtr overlapped);


        /* 
        BOOL __stdcall WinUsb_GetOverlappedResult(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   LPOVERLAPPED lpOverlapped,
          _Out_  LPDWORD lpNumberOfBytesTransferred,
          _In_   BOOL bWait
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetOverlappedResult(IntPtr interfaceHandle, IntPtr overlapped, out UInt32 numberOfBytesTransferred, bool wait);


        /* 
        BOOL __stdcall WinUsb_SetPipePolicy(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR PipeID,
          _In_  ULONG PolicyType,
          _In_  ULONG ValueLength,
          _In_  PVOID Value
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_SetPipePolicy(IntPtr interfaceHandle, byte pipeId, UInt32 policyType, UInt32 valueLength, UInt32[] value);

        /* 
        BOOL __stdcall WinUsb_GetPipePolicy(
          _In_     WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_     UCHAR PipeID,
          _In_     ULONG PolicyType,
          _Inout_  PULONG ValueLength,
          _Out_    PVOID Value
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetPipePolicy(IntPtr interfaceHandle, byte pipeId, UInt32 policyType, ref UInt32 valueLength, UInt32[] value);


        /* 
        BOOL __stdcall WinUsb_FlushPipe(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR PipeID
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_FlushPipe(IntPtr interfaceHandle, byte pipeId);


        /*
        BOOL __stdcall WinUsb_GetDescriptor(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   UCHAR DescriptorType,
          _In_   UCHAR Index,
          _In_   USHORT LanguageID,
          _Out_  PUCHAR Buffer,
          _In_   ULONG BufferLength,
          _Out_  PULONG LengthTransferred
        );
        */


        /*
        BOOL __stdcall WinUsb_QueryInterfaceSettings(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   UCHAR AlternateSettingNumber,
          _Out_  PUSB_INTERFACE_DESCRIPTOR UsbAltInterfaceDescriptor
        );
        */

        /*
        BOOL __stdcall WinUsb_GetCurrentAlternateSetting(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _Out_  PUCHAR AlternateSetting
        );
         */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetCurrentAlternateSetting(IntPtr interfaceHandle, out byte alternateSetting);


        /*
        BOOL __stdcall WinUsb_SetCurrentAlternateSetting(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR AlternateSetting
        );
        */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_SetCurrentAlternateSetting(IntPtr interfaceHandle, byte alternateSetting);
        #endregion

    }
    public struct NativeOverlapped
    {
        public IntPtr Internal;
        public IntPtr InternalHigh;
        public long Pointer; // On 32bit systems this is 32bit, but it's merged with an "Offset" field which is 64bit.
        public IntPtr Event;
    }
    public class Overlapped : IDisposable
    { 
    	public ManualResetEvent WaitEvent;
        public NativeOverlapped OverlappedStructShadow;
        public IntPtr OverlappedStruct;
        
        public Overlapped()
        {
            WaitEvent = new ManualResetEvent(false);
            OverlappedStructShadow = new NativeOverlapped();
            OverlappedStructShadow.Event = WaitEvent.SafeWaitHandle.DangerousGetHandle();

            OverlappedStruct = Marshal.AllocHGlobal(Marshal.SizeOf(OverlappedStructShadow));
            Marshal.StructureToPtr(OverlappedStructShadow, OverlappedStruct, false);
        }
        public void Dispose()
        {
            Marshal.FreeCoTaskMem(OverlappedStruct);
            
            WaitEvent.Close();
            //WaitEvent.Dispose(); //HACK Deaktiviert wegen sicherheitsebene
            GC.SuppressFinalize(this);
        }
    }
    public enum WinUsbPipePolicy
    {
        SHORT_PACKET_TERMINATE = 1,
        AUTO_CLEAR_STALL = 2,
        PIPE_TRANSFER_TIMEOUT = 3,
        IGNORE_SHORT_PACKETS = 4,
        ALLOW_PARTIAL_READS = 5,
        AUTO_FLUSH = 6,
        RAW_IO = 7,
        MAXIMUM_TRANSFER_SIZE = 8,
        RESET_PIPE_ON_RESUME = 9
    }
    #endregion
    
    #region WinUSBDevice
    public class WinUSBEnumeratedDevice
    {
        internal string DevicePath;
        internal EnumeratedDevice EnumeratedData;
        internal WinUSBEnumeratedDevice(EnumeratedDevice enumDev)
        {
            DevicePath = enumDev.DevicePath;
            EnumeratedData = enumDev;
            Match m = Regex.Match(DevicePath, @"vid_([\da-f]{4})");
            if (m.Success) { VendorID = Convert.ToUInt16(m.Groups[1].Value, 16); }
            m = Regex.Match(DevicePath, @"pid_([\da-f]{4})");
            if (m.Success) { ProductID = Convert.ToUInt16(m.Groups[1].Value, 16); }
            m = Regex.Match(DevicePath, @"mi_([\da-f]{2})");
            if (m.Success) { UsbInterface = Convert.ToByte(m.Groups[1].Value, 16); }
        }

        public string Path { get { return DevicePath; } }
        public UInt16 VendorID { get; private set; }
        public UInt16 ProductID { get; private set; }
        public Byte UsbInterface { get; private set; }
        public Guid InterfaceGuid { get { return EnumeratedData.InterfaceGuid; } }


        public override string ToString()
        {
            return string.Format("WinUSBEnumeratedDevice({0},{1})", DevicePath, InterfaceGuid);
        }
    }
    public class WinUSBDevice : IDisposable
    {
        public static IEnumerable<WinUSBEnumeratedDevice> EnumerateDevices(Guid deviceInterfaceGuid)
        {
            foreach (EnumeratedDevice devicePath in NativeMethods.EnumerateDevicesByInterface(deviceInterfaceGuid))
            {
                yield return new WinUSBEnumeratedDevice(devicePath);
            }
        }

        public static IEnumerable<WinUSBEnumeratedDevice> EnumerateAllDevices()
        {
            foreach (EnumeratedDevice devicePath in NativeMethods.EnumerateAllWinUsbDevices())
            {
                yield return new WinUSBEnumeratedDevice(devicePath);
            }
        }
        public delegate void NewDataCallback();

        string myDevicePath;
        SafeFileHandle deviceHandle;
        IntPtr WinusbHandle;

        internal bool Stopping = false;

        public WinUSBDevice(WinUSBEnumeratedDevice deviceInfo)
        {
            myDevicePath = deviceInfo.DevicePath;

            deviceHandle = NativeMethods.CreateFile(myDevicePath, NativeMethods.GENERIC_READ | NativeMethods.GENERIC_WRITE,
                NativeMethods.FILE_SHARE_READ | NativeMethods.FILE_SHARE_WRITE, IntPtr.Zero, NativeMethods.OPEN_EXISTING,
                NativeMethods.FILE_ATTRIBUTE_NORMAL | NativeMethods.FILE_FLAG_OVERLAPPED, IntPtr.Zero);

            if (deviceHandle.IsInvalid) {
            	System.Windows.Forms.MessageBox.Show("Winusb deviceHandle is invalid."); return;
                //throw new Exception("Could not create file. " + (new Win32Exception()).ToString());
            }

            if (!NativeMethods.WinUsb_Initialize(deviceHandle, out WinusbHandle)) {
                WinusbHandle = IntPtr.Zero;
                System.Windows.Forms.MessageBox.Show("Could not Initialize WinUSB."); return;
//                throw new Exception("Could not Initialize WinUSB. " + (new Win32Exception()).ToString());
            }


        }

        public byte AlternateSetting
        {
            get {
                byte alt;
                if (!NativeMethods.WinUsb_GetCurrentAlternateSetting(WinusbHandle, out alt))
                {
                    throw new Exception("GetCurrentAlternateSetting failed. " + (new Win32Exception()).ToString());
                }
                return alt;
            } set {
                if (!NativeMethods.WinUsb_SetCurrentAlternateSetting(WinusbHandle, value))
                {
                    throw new Exception("SetCurrentAlternateSetting failed. " + (new Win32Exception()).ToString());
                }
            }
        }

        public void Dispose()
        {
            Stopping = true;

            // Close handles which will cause background theads to stop working & exit.
            if (WinusbHandle != IntPtr.Zero)
            {
                NativeMethods.WinUsb_Free(WinusbHandle);
                WinusbHandle = IntPtr.Zero;
            }
            deviceHandle.Close();

            // Wait for pipe threads to quit
            foreach (BufferedPipeThread th in bufferedPipes.Values)
            {
                while (!th.Stopped) Thread.Sleep(5);
            }

            GC.SuppressFinalize(this);
        }

        public void Close()
        {
            Dispose();
        }

        public void FlushPipe(byte pipeId)
        {
            if (!NativeMethods.WinUsb_FlushPipe(WinusbHandle, pipeId))
            {
                throw new Exception("FlushPipe failed. " + (new Win32Exception()).ToString());
            }
        }

        public UInt32 GetPipePolicy(byte pipeId, WinUsbPipePolicy policyType)
        {
            UInt32[] data = new UInt32[1];
            UInt32 length = 4;

            if (!NativeMethods.WinUsb_GetPipePolicy(WinusbHandle, pipeId, (uint)policyType, ref length, data))
            {
                throw new Exception("GetPipePolicy failed. " + (new Win32Exception()).ToString());
            }

            return data[0];
        }

        public void SetPipePolicy(byte pipeId, WinUsbPipePolicy policyType, UInt32 newValue)
        {
            UInt32[] data = new UInt32[1];
            UInt32 length = 4;
            data[0] = newValue;

            if (!NativeMethods.WinUsb_SetPipePolicy(WinusbHandle, pipeId, (uint)policyType, length, data))
            {
                throw new Exception("SetPipePolicy failed. " + (new Win32Exception()).ToString());
            }
        }

        Dictionary<byte, BufferedPipeThread> bufferedPipes = new Dictionary<byte, BufferedPipeThread>();
        public void EnableBufferedRead(byte pipeId, int bufferCount, int multiPacketCount)
        {//byte pipeId, int bufferCount = 4, int multiPacketCount = 1
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                bufferedPipes.Add(pipeId, new BufferedPipeThread(this, pipeId, bufferCount, multiPacketCount));
            }
        }

        public void StopBufferedRead(byte pipeId)
        {

        }

        public void BufferedReadNotifyPipe(byte pipeId, NewDataCallback callback)
        {
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                throw new Exception("Pipe not enabled for buffered reads!");
            }
            bufferedPipes[pipeId].NewDataEvent += callback;
        }

        BufferedPipeThread GetInterface(byte pipeId, bool packetInterface)
        {
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                throw new Exception("Pipe not enabled for buffered reads!");
            }
            BufferedPipeThread th = bufferedPipes[pipeId];
            if (!th.InterfaceBound)
            {
                th.InterfaceBound = true;
                th.PacketInterface = packetInterface;
            }
            else
            {
                if (th.PacketInterface != packetInterface)
                {
                    string message = string.Format("Pipe is already bound as a {0} interface - cannot bind to both Packet and Byte interfaces",
                                                   packetInterface ? "Byte" : "Packet");
                    throw new Exception(message);
                }
            }
            return th;
        }
        public IPipeByteReader BufferedGetByteInterface(byte pipeId)
        {
            return GetInterface(pipeId, false);
        }

        public IPipePacketReader BufferedGetPacketInterface(byte pipeId)
        {
            return GetInterface(pipeId, true);
        }



        public byte[] BufferedReadPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).ReceiveBytes(byteCount);
        }

        public byte[] BufferedPeekPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).PeekBytes(byteCount);
        }

        public void BufferedSkipBytesPipe(byte pipeId, int byteCount)
        {
            BufferedGetByteInterface(pipeId).SkipBytes(byteCount);
        }

        public byte[] BufferedReadExactPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).ReceiveExactBytes(byteCount);
        }

        public int BufferedByteCountPipe(byte pipeId)
        {
            return BufferedGetByteInterface(pipeId).QueuedDataLength;
        }

        public UInt16[] ReadExactPipeU16(byte pipeId, int count)
        {
            int read = 0;
            UInt16[] accumulate = null;
            while (read < count)
            {
                UInt16[] data = ReadPipeU16(pipeId, count - read);
                if (data.Length == 0)
                {
                    // Timeout happened in ReadPipeU16.
                    throw new Exception("Timed out while trying to read data.");
                }
                if (data.Length == count) return data;
                if (accumulate == null)
                {
                    accumulate = new UInt16[count];
                }
                Array.Copy(data, 0, accumulate, read, data.Length);
                read += data.Length;
            }
            return accumulate;
        }

        public byte[] ReadExactPipe(byte pipeId, int byteCount)
        {
            int read = 0;
            byte[] accumulate = null;
            while (read < byteCount)
            {
                byte[] data = ReadPipe(pipeId, byteCount - read);
                if (data.Length == 0)
                {
                    // Timeout happened in ReadPipe.
                    return null;
//                    throw new Exception("Timed out while trying to read data.");
                }
                if (data.Length == byteCount) return data;
                if (accumulate == null)
                {
                    accumulate = new byte[byteCount];
                }
                Array.Copy(data, 0, accumulate, read, data.Length);
                read += data.Length;
            }
            return accumulate;
        }

        // basic synchronous read
        public UInt16[] ReadPipeU16(byte pipeId, int count)
        {

            byte[] data = new byte[count*2];

            UInt32 transferSize = 0;
            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, data, (uint)count*2, ref transferSize, IntPtr.Zero))
            {
                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
                {
                    // This was a pipe timeout. Return an empty byte array to indicate this case.
                    return new UInt16[0];
                }
                throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
            }

            UInt16[] newdata = new UInt16[transferSize / 2];
            for (int i = 0; i < (transferSize / 2); i++)
            {
                int v = BitConverter.ToInt16(data, i * 2);
                newdata[i] = (UInt16)v;
            }
            return newdata;

        }

        // basic synchronous read
        public byte[] ReadPipe(byte pipeId, int byteCount)
        {

            byte[] data = new byte[byteCount];

            UInt32 transferSize = 0;
            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, data, (uint)byteCount, ref transferSize, IntPtr.Zero))
            {
            	return new byte[0];
//                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
//                {
//                    // This was a pipe timeout. Return an empty byte array to indicate this case.
//                    
//                }
//                throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
            }

            byte[] newdata = new byte[transferSize];
            Array.Copy(data, newdata, transferSize);
            return newdata;

        }

        // Asynchronous read bits, only for use with buffered reader for now.
        internal void BeginReadPipe(byte pipeId, QueuedBuffer buffer)
        {
            buffer.Overlapped.WaitEvent.Reset();

            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, buffer.PinnedBuffer, (uint)buffer.BufferSize, IntPtr.Zero, buffer.Overlapped.OverlappedStruct))
            {
                if (Marshal.GetLastWin32Error() != NativeMethods.ERROR_IO_PENDING)
                {
                    throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
                }
            }
        }

        internal byte[] EndReadPipe(QueuedBuffer buf)
        {
            UInt32 transferSize;

            if (!NativeMethods.WinUsb_GetOverlappedResult(WinusbHandle, buf.Overlapped.OverlappedStruct, out transferSize, true))
            {
                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
                {
                    // This was a pipe timeout. Return an empty byte array to indicate this case.
                    //System.Diagnostics.Debug.WriteLine("Timed out");
                    return null;
                }
                throw new Exception("ReadPipe's overlapped result failed. " + (new Win32Exception()).ToString());
            }

            byte[] data = new byte[transferSize];
            Marshal.Copy(buf.PinnedBuffer, data, 0, (int)transferSize);
            return data;
        }


        // basic synchronous send.
        public void WritePipe(byte pipeId, byte[] pipeData)
        {

            int remainingbytes = pipeData.Length;
            while (remainingbytes > 0)
            {

                UInt32 transferSize = 0;
                if (!NativeMethods.WinUsb_WritePipe(WinusbHandle, pipeId, pipeData, (uint)pipeData.Length, ref transferSize, IntPtr.Zero))
                {
                    throw new Exception("WritePipe failed. " + (new Win32Exception()).ToString());
                }
                if (transferSize == pipeData.Length) return;

                remainingbytes -= (int)transferSize;

                // Need to retry. Copy the remaining data to a new buffer.
                byte[] data = new byte[remainingbytes];
                Array.Copy(pipeData, transferSize, data, 0, remainingbytes);

                pipeData = data;
            }
        }



        public void ControlTransferOut(byte requestType, byte request, UInt16 value, UInt16 index, byte[] data)
        {
            NativeMethods.WINUSB_SETUP_PACKET setupPacket = new NativeMethods.WINUSB_SETUP_PACKET();
            setupPacket.RequestType = (byte)(requestType | ControlDirectionOut);
            setupPacket.Request = request;
            setupPacket.Value = value;
            setupPacket.Index = index;
            if (data != null)
            {
                setupPacket.Length = (ushort)data.Length;
            }

            UInt32 actualLength = 0;

            if (!NativeMethods.WinUsb_ControlTransfer(WinusbHandle, setupPacket, data, setupPacket.Length, out actualLength, IntPtr.Zero))
            {
            	Dispose();
            	return;
            	//HACK just a try
                //throw new Exception("ControlTransfer failed. " + (new Win32Exception()).ToString());
            }
            
            if (data != null && actualLength != data.Length)
            {
                throw new Exception("Not all data transferred");
            }
        }

        public byte[] ControlTransferIn(byte requestType, byte request, UInt16 value, UInt16 index, UInt16 length)
        {
            NativeMethods.WINUSB_SETUP_PACKET setupPacket = new NativeMethods.WINUSB_SETUP_PACKET();
            setupPacket.RequestType = (byte)(requestType | ControlDirectionIn);
            setupPacket.Request = request;
            setupPacket.Value = value;
            setupPacket.Index = index;
            setupPacket.Length = length;

            byte[] output = new byte[length];
            UInt32 actualLength = 0;

            if(!NativeMethods.WinUsb_ControlTransfer(WinusbHandle, setupPacket, output, (uint)output.Length, out actualLength, IntPtr.Zero))
            {
            	Dispose();
            	return null;
            	//HACK just a try
//                throw new Exception("ControlTransfer failed. " + (new Win32Exception()).ToString());
            }

            if(actualLength != output.Length)
            {
                byte[] copyTo = new byte[actualLength];
                Array.Copy(output, copyTo, actualLength);
                output = copyTo;
            }
            return output;
        }

        const byte ControlDirectionOut = 0x00;
        const byte ControlDirectionIn = 0x80;

        public const byte ControlTypeStandard = 0x00;
        public const byte ControlTypeClass = 0x20;
        public const byte ControlTypeVendor = 0x40;

        public const byte ControlRecipientDevice = 0;
        public const byte ControlRecipientInterface = 1;
        public const byte ControlRecipientEndpoint = 2;
        public const byte ControlRecipientOther = 3;


    }
    internal class QueuedBuffer : IDisposable
    {
        public readonly int BufferSize;
        public Overlapped Overlapped;
        public IntPtr PinnedBuffer;
        public QueuedBuffer(int bufferSizeBytes)
        {
            BufferSize = bufferSizeBytes;
            Overlapped = new Overlapped();
            PinnedBuffer = Marshal.AllocHGlobal(BufferSize);
        }

        public void Dispose()
        {
            Overlapped.Dispose();
            Marshal.FreeHGlobal(PinnedBuffer);
            GC.SuppressFinalize(this);
        }

        public void Wait()
        {
            Overlapped.WaitEvent.WaitOne();
        }

        public bool Ready
        {
            get
            {
                return Overlapped.WaitEvent.WaitOne(0);
            }
        }
        
    }
    public interface IPipeByteReader
    {
        /// <summary>
        /// Receive a number of bytes from the incoming data stream.
        /// If there are not enough bytes available, only the available bytes will be returned.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] ReceiveBytes(int count);

        /// <summary>
        /// Receive a number of bytes from the incoming data stream, but don't remove them from the queue.
        /// If there are not enough bytes available, only the available bytes will be returned.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] PeekBytes(int count);

        /// <summary>
        /// Receive a specific number of bytes from the incoming data stream.
        /// This call will block until the requested bytes are available, or will eventually throw on timeout.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] ReceiveExactBytes(int count);

        /// <summary>
        /// Drop bytes from the incoming data stream without reading them.
        /// If you try to drop more bytes than are available, the buffer will be cleared.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to drop.</param>
        void SkipBytes(int count);

        /// <summary>
        /// Current number of bytes that are queued and available to read.
        /// </summary>
        int QueuedDataLength { get; }
    }
    public interface IPipePacketReader
    {
        /// <summary>
        /// Number of received packets that can be read.
        /// </summary>
        int QueuedPackets { get; }

        /// <summary>
        /// Retrieve the next packet, but do not remove it from the buffer.
        /// Warning: If you modify the returned array, the modifications will be present in future calls to Peek/Dequeue for this pacekt.
        /// </summary>
        /// <returns>The contents of the next packet in the receive queue</returns>
        byte[] PeekPacket();

        /// <summary>
        /// Retrieve the next packet from the receive queue
        /// </summary>
        /// <returns>The contents of the next packet in the receive queue</returns>
        byte[] DequeuePacket();
    }

    // Background thread to receive data from pipes.
    // Provides two data access mechanisms which are mutually exclusive: Packet level and byte level.
    internal class BufferedPipeThread : IPipeByteReader, IPipePacketReader
    {
        // Logic to enforce interface exclucivity is in WinUSBDevice
        public bool InterfaceBound; // Has the interface been bound?
        public bool PacketInterface; // Are we using the packet reader interface?


        Thread PipeThread;
        WinUSBDevice Device;
        byte DevicePipeId;

        private long TotalReceived;

        private int QueuedLength;
        private Queue<byte[]> ReceivedData;
        private int SkipFirstBytes;
        public bool Stopped = false;

        ManualResetEvent ReceiveTick;

        QueuedBuffer[] BufferList;
        Queue<QueuedBuffer> PendingBuffers;

        public BufferedPipeThread(WinUSBDevice dev, byte pipeId, int bufferCount, int multiPacketCount)
        {
            int maxTransferSize = (int)dev.GetPipePolicy(pipeId, WinUsbPipePolicy.MAXIMUM_TRANSFER_SIZE);

            int pipeSize = 512; // Todo: query pipe transfer size for 1:1 mapping to packets.
            int bufferSize = pipeSize * multiPacketCount;
            if (bufferSize > maxTransferSize) { bufferSize = maxTransferSize; }

            PendingBuffers = new Queue<QueuedBuffer>(bufferCount);
            BufferList = new QueuedBuffer[bufferCount];
            for (int i = 0; i < bufferCount;i++)
            {
                BufferList[i] = new QueuedBuffer(bufferSize);
            }

            EventConcurrency = new Semaphore(3, 3);
            Device = dev;
            DevicePipeId = pipeId;
            QueuedLength = 0;
            ReceivedData = new Queue<byte[]>();
            ReceiveTick = new ManualResetEvent(false);
            PipeThread = new Thread(ThreadFunc);
            PipeThread.IsBackground = true;

            //dev.SetPipePolicy(pipeId, WinUsbPipePolicy.PIPE_TRANSFER_TIMEOUT, 1000);

            // Start reading on all the buffers.
            foreach(QueuedBuffer qb in BufferList)
            {
                dev.BeginReadPipe(pipeId, qb);
                PendingBuffers.Enqueue(qb);
            }

            //dev.SetPipePolicy(pipeId, WinUsbPipePolicy.RAW_IO, 1);

            PipeThread.Start();
        }

        public long TotalReceivedBytes { get { return TotalReceived; } }

        //
        // Packet Reader members
        //

        public int QueuedPackets { get { lock (this) { return ReceivedData.Count; } } }

        public byte[] PeekPacket()
        {
            lock (this)
            {
                return ReceivedData.Peek();
            }
        }

        public byte[] DequeuePacket()
        {
            lock (this)
            {
                return ReceivedData.Dequeue();
            }
        }

        //
        // Byte Reader members
        //

        public int QueuedDataLength { get {  return QueuedLength;  } }

        // Only returns as many as it can.
        public byte[] ReceiveBytes(int count)
        {
            int queue = QueuedDataLength;
            if (queue < count) 
                count = queue;

            byte[] output = new byte[count];
            lock (this)
            {
                CopyReceiveBytes(output, 0, count);
            }
            return output;
        }

        // Only returns as many as it can.
        public byte[] PeekBytes(int count)
        {
            int queue = QueuedDataLength;
            if (queue < count)
                count = queue;

            byte[] output = new byte[count];
            lock (this)
            {
                CopyPeekBytes(output, 0, count);
            }
            return output;
        }

        public byte[] ReceiveExactBytes(int count)
        {
            byte[] output = new byte[count];
            if (QueuedDataLength >= count)
            {
                lock (this)
                {
                    CopyReceiveBytes(output, 0, count);
                }
                return output;
            }
            int failedcount = 0;
            int haveBytes = 0;
            while (haveBytes < count)
            {
                ReceiveTick.Reset();
                lock (this)
                {
                    int thisBytes = QueuedLength;

                    if(thisBytes == 0)
                    {
                        failedcount++;
                        if(failedcount > 3)
                        {
                            throw new Exception("Timed out waiting to receive bytes");
                        }
                    }
                    else
                    {
                        failedcount = 0;
                        if (thisBytes + haveBytes > count) thisBytes = count - haveBytes;
                        CopyReceiveBytes(output, haveBytes, thisBytes);
                    }
                    haveBytes += (int)thisBytes;
                }
                if(haveBytes < count)
                {
                    if (Stopped) throw new Exception("Not going to have enough bytes to complete request.");
                    ReceiveTick.WaitOne();
                }
            }
            return output;
        }

        public void SkipBytes(int count)
        {
            lock (this)
            {
                int queue = QueuedLength;
                if (queue < count)
                    throw new ArgumentException("count must be less than the data length");

                int copied = 0;
                while (copied < count)
                {
                    byte[] firstData = ReceivedData.Peek();
                    int available = firstData.Length - SkipFirstBytes;
                    int toCopy = count - copied;
                    if (toCopy > available) toCopy = available;

                    if (toCopy == available)
                    {
                        ReceivedData.Dequeue();
                        SkipFirstBytes = 0;
                    }
                    else
                    {
                        SkipFirstBytes += toCopy;
                    }

                    copied += toCopy;
                    QueuedLength -= toCopy;
                }
            }
        }

        //
        // Internal functionality
        //

        // Must be called under lock with enough bytes in the buffer.
        void CopyReceiveBytes(byte[] target, int start, int count)
        {
            int copied = 0;
            while(copied < count)
            {
                byte[] firstData = ReceivedData.Peek();
                int available = firstData.Length - SkipFirstBytes;
                int toCopy = count - copied;
                if (toCopy > available) toCopy = available;

                Array.Copy(firstData, SkipFirstBytes, target, start, toCopy); 

                if(toCopy == available)
                {
                    ReceivedData.Dequeue();
                    SkipFirstBytes = 0;
                }
                else
                {
                    SkipFirstBytes += toCopy;
                }

                copied += toCopy;
                start += toCopy;
                QueuedLength -= toCopy;
            }
        }

        // Must be called under lock with enough bytes in the buffer.
        void CopyPeekBytes(byte[] target, int start, int count)
        {
            int copied = 0;
            int skipBytes = SkipFirstBytes;

            foreach(byte[] firstData in ReceivedData)
            {
                int available = firstData.Length - skipBytes;
                int toCopy = count - copied;
                if (toCopy > available) toCopy = available;

                Array.Copy(firstData, skipBytes, target, start, toCopy);

                skipBytes = 0;

                copied += toCopy;
                start += toCopy;

                if (copied >= count)
                {
                    break;
                }
            }
        }




        void ThreadFunc(object context)
        {
            Queue<byte[]> receivedData = new Queue<byte[]>(BufferList.Length);

            while(true)
            {
                if (Device.Stopping)
                    break;

                try
                {
                    PendingBuffers.Peek().Wait();
                    // Process a large group of received buffers in a batch, if available.
                    int n = 0;
                    try
                    {
                        while (n < BufferList.Length)
                        {
                            QueuedBuffer buf = PendingBuffers.Peek();
                            if (n == 0 || buf.Ready)
                            {
                                byte[] data = Device.EndReadPipe(buf);
                                PendingBuffers.Dequeue();
                                if (data != null)
                                {   // null is a timeout condition.
                                    receivedData.Enqueue(data);
                                }
                                Device.BeginReadPipe(DevicePipeId, buf);
                                // Todo: If this operation fails during normal operation, the buffer is lost from rotation.
                                // Should never happen during normal operation, but should confirm and mitigate if it's possible.
                                PendingBuffers.Enqueue(buf);

                            }
                            n++;
                        }
                    }
                    finally
                    {
                        // Unless we're exiting, ensure we always indicate the data, even if some operation failed.
                        if(!Device.Stopping && receivedData.Count > 0)
                        {
                            lock (this)
                            {
                                foreach (byte[] data in receivedData)
                                {
                                    ReceivedData.Enqueue(data);
                                    QueuedLength += data.Length;
                                    TotalReceived += data.Length;
                                }
                            }
                            ThreadPool.QueueUserWorkItem(RaiseNewData);
                            receivedData.Clear();
                        }
                    }
                }
                catch(Exception ex)
                {
                    System.Diagnostics.Debug.Print("Should not happen: Exception in background thread. {0}", ex.ToString());
                    Thread.Sleep(15);
                }

                ReceiveTick.Set();

            }
            Stopped = true;
        }

        public event WinUSBDevice.NewDataCallback NewDataEvent;

        Semaphore EventConcurrency;

        void RaiseNewData(object context)
        {
            WinUSBDevice.NewDataCallback cb = NewDataEvent;
            if (cb != null)
            {
                if(EventConcurrency.WaitOne(0)) // Prevent requests from stacking up; Don't issue new events if there are several in flight
                {
                    try
                    {
                        cb();
                    }
                    finally
                    {
                        EventConcurrency.Release();
                    }

                }
            }
        }

    }
    #endregion
    
    #region UnsafeBitmap
    public unsafe class UnsafeBitmap
	{
		Bitmap bitmap;

		// three elements used for MakeGreyUnsafe
		int width;
		BitmapData bitmapData = null;
		Byte* pBase = null;
		public UnsafeBitmap(Bitmap bitmap)
		{
			this.bitmap = new Bitmap(bitmap);
		}
		public UnsafeBitmap(int width, int height)
		{
			this.bitmap = new Bitmap(width, height, PixelFormat.Format24bppRgb);
		}
		public void Dispose()
		{
			bitmap.Dispose();
		}
		public Bitmap Bitmap
		{
			get
			{
				return(bitmap);
			}
		}
		private Point PixelSize
		{
			get
			{
				GraphicsUnit unit = GraphicsUnit.Pixel;
				RectangleF bounds = bitmap.GetBounds(ref unit);

				return new Point((int) bounds.Width, (int) bounds.Height);
			}
		}
		public void LockBitmap()
		{
			GraphicsUnit unit = GraphicsUnit.Pixel;
			RectangleF boundsF = bitmap.GetBounds(ref unit);
			Rectangle bounds = new Rectangle((int) boundsF.X,
			                                 (int) boundsF.Y,
			                                 (int) boundsF.Width,
			                                 (int) boundsF.Height);
			
			// Figure out the number of bytes in a row
			// This is rounded up to be a multiple of 4
			// bytes, since a scan line in an image must always be a multiple of 4 bytes
			// in length.
			width = (int) boundsF.Width * sizeof(PixelData);
			if (width % 4 != 0)
			{
				width = 4 * (width / 4 + 1);
			}
			bitmapData =
				bitmap.LockBits(bounds, ImageLockMode.ReadWrite, PixelFormat.Format24bppRgb);
			pBase = (Byte*) bitmapData.Scan0.ToPointer();
		}
		public PixelData GetPixel(int x, int y)
		{
			//x=337 / y=489
			try {
				PixelData returnValue = *PixelAt(x, y);
				return returnValue;
			} catch (Exception) {
				PixelData returnValue = *PixelAt(1, 1);
				return returnValue;
			}
			
		}
		public void SetPixel(int x, int y, PixelData colour)
		{
			PixelData* pixel = PixelAt(x, y);
			*pixel = colour;
		}
		public void UnlockBitmap()
		{
			bitmap.UnlockBits(bitmapData);
			bitmapData = null;
			pBase = null;
		}
		public PixelData* PixelAt(int x, int y)
		{
			return (PixelData*)(pBase + y * width + x * sizeof(PixelData));
		}
	}
	public struct PixelData
	{
		public byte blue;
		public byte green;
		public byte red;
	}
    #endregion
    
    #region _Unused
//    public class RgbButton
//    {
//        public static IEnumerable<WinUSBEnumeratedDevice> Enumerate()
//        {
//            return WinUSBDevice.EnumerateDevices(new Guid("60d2a26c-1d22-4538-b7d5-337b2f07fff1"));
//        }
//
//        const byte OUT_PIPE = 0x03;
//        const byte IN_PIPE = 0x83;
//        const int ButtonThreshold = 0x70;
//
//        WinUSBDevice BaseDevice;
//        public RGBColor[] ButtonColors;
//        public int[] ButtonValues;
//        public bool[] ButtonPressed;
//        public long DataCount;
//
//        public RgbButton(WinUSBEnumeratedDevice dev)
//        {
//            BaseDevice = new WinUSBDevice(dev);
//
//            ButtonColors = new RGBColor[4];
//            ButtonValues = new int[4];
//            ButtonPressed = new bool[4];
//
//            BaseDevice.EnableBufferedRead(IN_PIPE,4,1);
//            BaseDevice.BufferedReadNotifyPipe(IN_PIPE, NewDataCallback);
//            EnableButtonData();
//        }
//
//        public void Close()
//        {
//            BaseDevice.Close();
//            BaseDevice = null;
//        }
//
//        void NewDataCallback()
//        {
//            lock (this) // Prevent concurrent execution
//            {
//                bool newData = false;
//                bool badData;
//
//                while (BaseDevice.BufferedByteCountPipe(IN_PIPE) >= 5)
//                {
//                    badData = false;
//                    byte[] data = BaseDevice.BufferedPeekPipe(IN_PIPE, 5);
//                    if(data[0] != 0xFF) {
//                        badData = true;
//                    }
//                    for (int i = 0; i < 4; i++) {
//                        if(data[i + 1] >= 0x80)
//                        {
//                            // This is also a bad data (truncated) message, this can happen if the host falls behind.
//                            badData = true;
//                            break;
//                        }
//                    }
//                    if(badData) {
//                        // Bad data. Try again next byte.
//                        BaseDevice.BufferedSkipBytesPipe(IN_PIPE, 1);
//                        continue;
//                    }
//
//                    // This looks like a button message
//                    for(int i=0;i<4;i++) {
//                        ButtonValues[i] = data[i + 1];
//                        ButtonPressed[i] = ButtonValues[i] < ButtonThreshold;
//                    }
//                    newData = true;
//                    DataCount++;
//                    BaseDevice.BufferedSkipBytesPipe(IN_PIPE, 5);
//                }
//
//                if(newData) {
//                    // Provide notification.
//                }
//            }
//        }
//
//        public void SendButtonColors()
//        {
//            byte[] command = new byte[13];
//            command[0] = (byte)'L';
//
//            int n = 1;
//            foreach(RGBColor c in ButtonColors) {
//                command[n++] = c.RByte;
//                command[n++] = c.GByte;
//                command[n++] = c.BByte;
//            }
//            BaseDevice.WritePipe(OUT_PIPE, command);
//        }
//
//        void SendByteCommand(byte b)
//        {
//            byte[] command = new byte[1];
//            command[0] = b;
//            BaseDevice.WritePipe(OUT_PIPE, command);
//        }
//        void EnableButtonData()
//        {
//            SendByteCommand((byte)'B');
//        }
//        void DisableButtonData()
//        {
//            SendByteCommand((byte)'X');
//        }
//        public void EnterProgrammingMode()
//        {
//            SendByteCommand((byte)'P');
//        }
//    }
//    public struct RGBColor
//    {
//        public double R, G, B;
//
//        byte ConvertValue(double source)
//        {
//            if (source < 0) source = 0;
//            if (source > 1) source = 1;
//            return (byte)(source * 255);
//        }
//        public byte RByte { get { return ConvertValue(R); } }
//        public byte GByte { get { return ConvertValue(G); } }
//        public byte BByte { get { return ConvertValue(B); } }
//
//        public RGBColor(double r, double g, double b)
//        {
//            R = r; G = g; B = b;
//        }
//
//        public static RGBColor FromHSV(double h, double s, double v)
//        {
//            double r, g, b;
//            r = g = b = 0;
//            if (h < 0.5) r = 1 - h * 3; else r = h * 3 - 2;
//            g = 1 - Math.Abs(h * 3 - 1);
//            b = 1 - Math.Abs(h * 3 - 2);
//            if (r > 1) r = 1;
//            if (r < 0) r = 0;
//            if (g > 1) g = 1;
//            if (g < 0) g = 0;
//            if (b > 1) b = 1;
//            if (b < 0) b = 0;
//
//            r = (r * s + (1 - s)) * v;
//            g = (g * s + (1 - s)) * v;
//            b = (b * s + (1 - s)) * v;
//
//            return new RGBColor(r, g, b);
//        }
//    }
    
//    public class CalibratedThermalFrame
//    {
//        public readonly int Width, Height;
//        public readonly UInt16[] PixelData;
//        public UInt16 MinValue;
//        public UInt16 MaxValue;
//        
//        internal CalibratedThermalFrame(UInt16[] data)
//        {
//            Width = 208;
//            Height = 156;
//            PixelData = data;
//
//            ushort min = 0xFFFF;
//            ushort max = 0x0000;
//            foreach(ushort v in PixelData)
//            {
//                //if (v < 2000) continue;
//                //if (v > 49152) continue;
//                // 14K to 32K is valid for min/max
//                if (v < 14336) continue;
//                if (v > 32767) continue;
//                if (v < min) min = v;
//                if (v > max) max = v;
//            }
//            MinValue = min;
//            MaxValue = max;
//        }
//        // Needed to override Min/Max (for external adjustment).
//        public void SetMinMax(UInt16 min, UInt16 max)
//        {
//            MinValue = min;
//            MaxValue = max;
//        }
//    }    

//    public class HackRF
//    {
//        const byte EP_RX = 0x81; // IN 1
//        const byte EP_TX = 0x02; // OUT 2
//
//        WinUSBDevice Device;
//        IPipePacketReader RxPipeReader;
//
//        public static IEnumerable<WinUSBEnumeratedDevice> Enumerate()
//        {
//            foreach (WinUSBEnumeratedDevice dev in WinUSBDevice.EnumerateAllDevices())
//            {
//                // HackRF One
//                if (dev.VendorID == 0x1d50 && dev.ProductID == 0x6089 && dev.UsbInterface == 0)
//                {
//                    yield return dev;
//                }
//                // HackRF Jawbreaker
//                if (dev.VendorID == 0x1d50 && dev.ProductID == 0x604b && dev.UsbInterface == 0)
//                {
//                    yield return dev;
//                }
//            }
//        }
//        public HackRF(WinUSBEnumeratedDevice dev)
//        {
//            Device = new WinUSBDevice(dev);
//
//            // Set a bunch of sane defaults.
//            SetSampleRate(10000000,1); // 10MHz
//            SetFilterBandwidth(10000000);
//            SetLnaGain(8);
//            SetVgaGain(20);
//            SetTxVgaGain(0);
//        }
//        public void Close()
//        {
//            Device.Close();
//            Device = null;
//        }
//        void RxDataCallback()
//        {
//            lock (this)
//            {
//                while (RxPipeReader.QueuedPackets > 0)
//                {
//                    int len = RxPipeReader.DequeuePacket().Length;
//                    BytesEaten += len;
//                    if (!EatenHistogram.ContainsKey(len)) { EatenHistogram.Add(len, 0); }
//                    EatenHistogram[len]++;
//                    PacketsEaten++;
//                }
//            }
//        }
//
//        public int PacketsEaten; // debug
//        public long BytesEaten;
//        public Dictionary<int, int> EatenHistogram = new Dictionary<int, int>();
//
//        enum DeviceRequest
//        {
//            SetTransceiverMode = 1,
//            SetSampleRate = 6,
//            SetFilterBandwidth = 7,
//            VersionStringRead = 15,
//            SetFrequency = 16,
//            AmpEnable = 17,
//            SetLnaGain = 19,
//            SetVgaGain = 20,
//            SetTxVgaGain = 21
//        }
//        enum TransceiverMode
//        {
//            Off = 0,
//            Receive = 1,
//            Transmit = 2,
//        }
//
//        byte[] VendorRequestIn(DeviceRequest request, ushort value, ushort index, ushort length)
//        {
//            byte requestType = WinUSBDevice.ControlRecipientDevice | WinUSBDevice.ControlTypeVendor;
//
//            return Device.ControlTransferIn(requestType, (byte)request, value, index, length);
//        }
//        void VendorRequestOut(DeviceRequest request, ushort value, ushort index, byte[] data)
//        {
//            byte requestType = WinUSBDevice.ControlRecipientDevice | WinUSBDevice.ControlTypeVendor;
//            Device.ControlTransferOut(requestType, (byte)request, value, index, data);
//        }
//
//        void SetTransceiverMode(TransceiverMode mode)
//        {
//            VendorRequestOut(DeviceRequest.SetTransceiverMode, (ushort)mode, 0, null);
//        }
//
//        public string ReadVersion()
//        {
//            byte[] data = VendorRequestIn(DeviceRequest.VersionStringRead, 0, 0, 255);
//            return new string(data.Select(b => (char)b).ToArray());
//        }
//        public void ModeOff()
//        {
//            SetTransceiverMode(TransceiverMode.Off);
//        }
//        public void ModeReceive()
//        {
//            SetTransceiverMode(TransceiverMode.Receive);
//
//            Device.EnableBufferedRead(EP_RX, 4,64);
//            RxPipeReader = Device.BufferedGetPacketInterface(EP_RX);
//            Device.BufferedReadNotifyPipe(EP_RX, RxDataCallback);
//        }
//        public void ModeTransmit()
//        {
//            SetTransceiverMode(TransceiverMode.Transmit);
//        }
//        
//        public void SetSampleRate(uint integerFrequency, uint divider)
//        { //uint integerFrequency, uint divider = 1
//            byte[] parameters = new byte[8];
//
//            byte[] value = BitConverter.GetBytes(integerFrequency);
//            Array.Copy(value, parameters, 4);
//
//            value = BitConverter.GetBytes(divider);
//            Array.Copy(value, 0, parameters, 4, 4);
//
//            VendorRequestOut(DeviceRequest.SetSampleRate, 0, 0, parameters);
//        }
//        public uint SetFilterBandwidth(uint requestedFilterBandwidth)
//        {
//            uint[] filterValues = { 1750000, 2250000, 3500000, 5000000,
//                                    5500000, 6000000, 7000000, 8000000,
//                                    9000000,10000000,12000000,14000000,
//                                   15000000,20000000,24000000,28000000 };
//
//            uint actualBandwidth = filterValues.TakeWhile(value => value < requestedFilterBandwidth).LastOrDefault();
//            if (actualBandwidth == 0) 
//                actualBandwidth = filterValues[0];
//
//            VendorRequestOut(DeviceRequest.SetFilterBandwidth,(ushort)(actualBandwidth&0xFFFF), (ushort)(actualBandwidth>>16), null);
//
//            return actualBandwidth;
//        }
//        public void SetFrequency(ulong frequency)
//        {
//            uint mhz = (uint)(frequency / 1000000);
//            uint hz = (uint)(frequency % 1000000);
//
//            byte[] parameters = new byte[8];
//
//            byte[] value = BitConverter.GetBytes(mhz);
//            Array.Copy(value, parameters, 4);
//
//            value = BitConverter.GetBytes(hz);
//            Array.Copy(value, 0, parameters, 4, 4);
//
//            VendorRequestOut(DeviceRequest.SetFrequency, 0, 0, parameters);
//        }
//        public void SetLnaGain(uint gaindB)
//        {
//            if (gaindB > 40) throw new ArgumentOutOfRangeException("gaindB", "Lna Gain should be in the range 0-40");
//            byte[] check = VendorRequestIn(DeviceRequest.SetLnaGain, 0, (ushort)gaindB, 1);
//        }
//        public void SetVgaGain(uint gaindB)
//        {
//            if (gaindB > 62) throw new ArgumentOutOfRangeException("gaindB","Vga Gain should be in the range 0-62");
//            byte[] check = VendorRequestIn(DeviceRequest.SetVgaGain, 0, (ushort)gaindB, 1);            
//        }
//        public void SetTxVgaGain(uint gaindB)
//        {
//            if (gaindB > 47) throw new ArgumentOutOfRangeException("gaindB", "TxVga Gain should be in the range 0-47");
//            byte[] check = VendorRequestIn(DeviceRequest.SetTxVgaGain, 0, (ushort)gaindB, 1);    
//        }
//    }

//    public class Fadecandy : IDisposable
//    {
//    	const byte DataPipe = 0x01; // OUT 1
//    	public WinUSBDevice BaseDevice;
//    	public RGBColor[] Pixels;
//    	
//        public static IEnumerable<WinUSBEnumeratedDevice> Enumerate()
//        {
//            foreach(WinUSBEnumeratedDevice dev in WinUSBDevice.EnumerateAllDevices())
//            {
//                if (dev.VendorID == 0x1d50 && dev.ProductID == 0x607A && dev.UsbInterface == 0)
//                {
//                    yield return dev;
//                }
//            }
//        }
//        public Fadecandy(WinUSBEnumeratedDevice dev)
//        {
//            BaseDevice = new WinUSBDevice(dev);
//            Pixels = new RGBColor[512];
//            Initialize();
//        }
//        public void Dispose()
//        {
//            if (BaseDevice != null)
//            {
//                BaseDevice.Dispose();
//                BaseDevice = null;
//            }
//            GC.SuppressFinalize(this);
//        }
//        public void Close()
//        {
//            BaseDevice.Close();
//            BaseDevice = null;
//        }
//        public void FlushAll()
//        {
//            FlushRange(0, 512);
//        }
//        public void FlushRange(int start, int count)
//        {
//            if (start < 0 || start > 511) throw new ArgumentException("start");
//            if (count < 0 || (start + count) > 512) throw new ArgumentException("count");
//            const int pixelsPerChunk = 21;
//
//            int firstChunk = (start / pixelsPerChunk);
//            int lastChunk = ((start + count - 1) / pixelsPerChunk);
//
//            byte[] data = new byte[64];
//            for (int chunk = firstChunk; chunk <= lastChunk; chunk++)
//            {
//                int offset = chunk * pixelsPerChunk;
//                data[0] = ControlByte(0, chunk == lastChunk, chunk);
//                for (int i = 0; i < pixelsPerChunk; i++)
//                {
//                    if (i + offset > 511) continue;
//                    data[1 + i * 3] = Pixels[i + offset].RByte; // Some weird LEDs I have have R/G switched. Odd.
//                    data[2 + i * 3] = Pixels[i + offset].GByte;
//                    data[3 + i * 3] = Pixels[i + offset].BByte;
//                }
//                BaseDevice.WritePipe(DataPipe, data);
//            }
//        }
//        public void Initialize()
//        {
//            double gammaCorrection = 1.6;
//            // compute basic uniform gamma table for r/g/b
//
//            const int lutEntries = 257;
//            const int lutTotalEntries = lutEntries * 3;
//            UInt16[] lutValues = new UInt16[lutTotalEntries];
//
//            for(int i=0;i<lutEntries;i++)
//            {
//                double r, g, b;
//                r = g = b = Math.Pow((double)i / (lutEntries-1), gammaCorrection) * 65535;
//                lutValues[i] = (UInt16)r;
//                lutValues[i+lutEntries] = (UInt16)g;
//                lutValues[i+lutEntries*2] = (UInt16)b;
//            }
//
//            // Send LUT 31 entries at a time.
//            byte[] data = new byte[64];
//
//            int blockIndex = 0;
//            int lutIndex = 0;
//            while (lutIndex < lutTotalEntries)
//            {
//                bool lastChunk = false;
//                int lutCount = (lutTotalEntries - lutIndex);
//                if (lutCount > 31)
//                    lutCount = 31;
//
//                int nextIndex = lutIndex + lutCount;
//                if (nextIndex == lutTotalEntries)
//                    lastChunk = true;
//
//                data[0] = ControlByte(1, lastChunk, blockIndex);
//
//                for (int i = 0; i < lutCount; i++)
//                {
//                    data[i * 2 + 2] = (byte)(lutValues[lutIndex + i] & 0xFF);
//                    data[i * 2 + 3] = (byte)((lutValues[lutIndex + i] >> 8) & 0xFF);
//                }
//
//                BaseDevice.WritePipe(DataPipe, data);
//
//                blockIndex++;
//                lutIndex = nextIndex;
//            }
//
//        }
//        byte ControlByte(int type, bool final, int index)
//        {
//            if (type < 0 || type > 3) throw new ArgumentException("type");
//            if (index < 0 || index > 31) throw new ArgumentException("index");
//            byte output = (byte)((type << 6) | index);
//            if (final) output |= 0x20;
//            return output;
//        }
//        public void SendConfiguration(bool enableDithering, bool enableKeyframeInterpolation, bool manualLedControl, bool ledValue, bool reservedMode)
//        {
//        	//bool enableDithering = true, bool enableKeyframeInterpolation = true, bool manualLedControl = false, bool ledValue = false, bool reservedMode = false
//            byte[] data = new byte[64];
//
//            data[0] = ControlByte(2,false,0);
//
//            if (!enableDithering) 
//                data[1] |= 0x01;
//
//            if (enableKeyframeInterpolation)
//                data[1] |= 0x02;
//
//            if (manualLedControl)
//                data[1] |= 0x04;
//
//            if (ledValue)
//                data[1] |= 0x08;
//
//            if (reservedMode)
//                data[1] |= 0x10;
//
//
//            BaseDevice.WritePipe(DataPipe, data);
//        }
//    }	
    #endregion
    
    
}