﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading;

namespace FlashLibStm32Uart
{
    public class FlashStm32
    {
        //based on https://github.com/MightyDevices/STBootLib

        SerialPort SP;
        //SemaphoreSlim sem;
        List<STCmds> Commands = new List<STCmds>();
        bool init = false;
        public bool Cancel = false;
        public event Action<float> EventReportProgress;
        public event Action<string> EventReportMessage;
        public uint BaseAddress = 0x08000000;
        public uint Address;
        public uint PageSize;
        public int TimeoutSeconds = 1;
        string SerialTransmissionMessage = "";
        public bool IsDebugMessage = false;

        public string Version;
        public int ProductID;
        public int RetryCntErrase = 0;
        public int RetryCntWrite = 0;
        public bool SingleByteMode = false;
        public bool PortStayOpen = false;
        public bool IsErrorExisting = false;
        public byte[] FileBuffer;
        public byte[] FileBufferRead;
        StringBuilder sbErrors = new StringBuilder();

        public FlashStm32(SerialPort sP) {
            SP = sP;
            //sem = new SemaphoreSlim(1);
            Address = BaseAddress;
            init = true;
        }
        void HandleError(Exception ex) {
            sbErrors.AppendLine(ex.Message);
            IsErrorExisting = true;
        }
        public string GetErrors() {
            string output = sbErrors.ToString();
            sbErrors.Clear();
            IsErrorExisting = false;
            return output;
        }
        void CheckInit() {
            if (init) {
                return;
            }
            throw new Exception("Not Initialised, run Init(...).");
        }
        byte ComputeChecksum(byte[] data, int offset, int count) {
            byte xor = 0xff;
            for (int i = offset; i < count + offset; i++) {
                xor ^= data[i];
            }
            return xor;
        }
        void OnEventReportProgress(float percent) {
            if (EventReportProgress != null) {
                EventReportProgress(percent);
            }
        }
        void OnEventReportMessage(string message, bool debugMessage) {
            if (!IsDebugMessage && debugMessage) {
                return;
            }
            if (EventReportMessage != null) {
                EventReportMessage(message);
            }
        }
        public bool VerifyBuffers() {
            int maxCompare = FileBuffer.Length;
            if (FileBufferRead.Length < maxCompare) {
                maxCompare = FileBufferRead.Length;
            }
            if (FileBufferRead.Length == FileBuffer.Length) {
                OnEventReportMessage("Verify buffers: " + maxCompare, false);
            }
            else {
                OnEventReportMessage($"Verify '{maxCompare}' from buffer write: {FileBuffer.Length}", false);
                OnEventReportMessage($"Verify '{maxCompare}' from buffer read: {FileBufferRead.Length}", false);
            }
            
            int diffCount = FileBufferRead.Length - FileBuffer.Length;
            if (diffCount < 0) {
                diffCount = 0 - diffCount;
            }
            int compared = 0;
            for (int i = 0; i < maxCompare; i++) {
                compared++;
                if (FileBuffer[i] != FileBufferRead[i]) {
                    diffCount++;
                }
            }
            OnEventReportMessage($"Verified '{compared}' with final byte diff: {diffCount}", false);
            if (diffCount == 0) {
                return true;
            }
            return false;
        }

        public void SerialPortOpen() {
            CheckInit();
            if (SP.IsOpen) {
                return;
            }
            SP.Open();
            SP.ReadTimeout = 3000;

            SP.DiscardInBuffer();
            SP.DiscardOutBuffer();
        }
        public void SerialPortClose(bool forceClose) {
            CheckInit();
            if (!forceClose && PortStayOpen) {
                return;
            }
            SP.Close();
        }
        
        public bool StepInit() {
            CheckInit();
            try {
                SerialPortOpen();

                SP.DiscardInBuffer();
                SP.DiscardOutBuffer();
                SP.Write(new byte[] { 127 }, 0, 1);
                Thread.Sleep(200);
                if (SP.BytesToRead != 0) {
                    byte[] buff = new byte[1];
                    SP.Read(buff, 0, 1);
                    if (buff[0] == 121 || buff[0] == 31) {
                        return true;
                    }
                }
                SP.Write(new byte[] { 127 }, 0, 1);
                Thread.Sleep(200);
                if (SP.BytesToRead != 0) {
                    byte[] buff = new byte[1];
                    SP.Read(buff, 0, 1);
                    if (buff[0] == 121 || buff[0] == 31) {
                        return true;
                    }
                } else { 
                    throw new Exception("No response from Device."); 
                }
            }
            catch (Exception ex) {
                HandleError(ex);
            }
            SerialPortClose(false);
            return false;
        }
        public void StepGetVersion() {
            SerialPortOpen();

            var tx = new byte[2];// { 0, 255 };

            tx[0] = (byte)STCmds.GET;
            tx[1] = ComputeChecksum(tx, 0, 1);

            int nbytes;
            var tmp = new byte[15];

            try {
                SerialWrite(tx, 0, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK) {
                    throw new Exception($"Command GET Rejected.<{tmp[0]}>"); 
                }
                /* wait for number of bytes */
                SerialRead(tmp, 0, 1);
                /* assign number of bytes that will follow (add for acks) */
                nbytes = tmp[0] + 2;
                /* nbytes must be equal to 13 for stm32 products */
                if (nbytes != 13) {
                    throw new Exception($"Invalid length ('{nbytes}' != 13)");
                }

                /* prepare buffer */
                tmp = new byte[nbytes];
                /* receive response */
                SerialRead(tmp, 0, tmp.Length);
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }

            /* store version information */
            Version = (tmp[0] >> 4).ToString() + "." + (tmp[0] & 0xf).ToString();

            Commands.Clear();
            for (int i = 2; i < tmp.Length - 1; i++) {
                Commands.Add((STCmds)tmp[i]);
            }
            SerialPortClose(false);
        }
        public void StepGetProductID() {
            SerialPortOpen();
            /* command word */
            var tx = new byte[2];
            /* temporary storage for response bytes */
            var tmp = new byte[1];
            /* numbe or response bytes */
            int nbytes;

            /* store code */
            tx[0] = (byte)STCmds.GET_ID;
            /* set checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);

                ///* wait for response code */
                SerialRead(tmp, 0, 1);
                if (tmp.Length == 0) {
                    throw new Exception("no Response");
                }
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                SerialRead(tmp, 0, 1);
                /* assign number of bytes that will follow (add for acks) */
                nbytes = tmp[0] + 2;
                /* nbytes must be equal to 3 (+ ACK + Checksum) for stm32 products */
                if (nbytes != 3)
                    throw new Exception("Invalid length (" + tmp.Length + ")");
                tmp = new byte[nbytes];
                /* receive response */
                SerialRead(tmp, 0, tmp.Length);
                ProductID = (tmp[0] << 8 | tmp[1]);

            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }

            SerialPortClose(false);
        }
        public void StepReadBinary(string filePath) {
            FileBuffer = null;
            if (!File.Exists(filePath)) {
                throw new Exception("File not found: " + filePath);
            }
            FileStream fs = new FileStream(filePath, FileMode.Open);
            FileBuffer = new byte[fs.Length];
            fs.Read(FileBuffer, 0, (int)fs.Length);
            fs.Close();
        }
        public void StepExecute(uint address) {
            if (!Commands.Contains(STCmds.GO)) {
                throw new Exception("Command not supported"); 
            }
            /* command word */
            var tx = new byte[7];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.GO;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* store address */
            tx[2] = (byte)((address >> 24) & 0xff);
            tx[3] = (byte)((address >> 16) & 0xff);
            tx[4] = (byte)((address >> 8) & 0xff);
            tx[5] = (byte)(address & 0xff);
            /* address checksum (needs to be not negated. why? because ST! 
             * that's why. */
            tx[6] = (byte)~ComputeChecksum(tx, 2, 4);

            /* try to send command and wait for response */
            try {
                SerialWrite(tx, 0, 2);

                SerialRead(tmp, 0, 1);
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");
                SerialWrite(tx, 2, 5);

                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Address Rejected");
                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
        public void StepErase(bool globalErase) {
            Cancel = false;
            RetryCntErrase = 0;
            if (globalErase) {
                for (int i = 5; i >= 0; i--) {
                    if (Cancel) { return; }
                    try {
                        /* 'classic' erase operation supported? */
                        if (Commands.Contains(STCmds.ERASE)) {
                            EraseGlobal();
                            return;
                        }
                        else if (Commands.Contains(STCmds.EXT_ERASE)) {
                            EraseSpecial(STExtendedEraseMode.GLOBAL);
                            return;
                        }
                        else {
                            throw new Exception("Command not supported");
                        }
                    }
                    catch (Exception ex) {
                        if (RetryCntErrase < 5) { //try again
                            OnEventReportMessage($"Retry '{RetryCntErrase}' EraseGlobal '{i}': {ex.Message}", true);
                            RetryCntErrase++;
                            i++;
                            continue;
                        }
                        HandleError(ex);
                        throw ex;
                    }
                }
                return;
            }
            /* erase operation */
            int binLen = FileBuffer.Length;
            for (uint i = 0; i < binLen; i += PageSize) {
                try {
                    /* erase page */
                    ErasePage((i + Address - BaseAddress) / PageSize);
                    /* update progress bar */
                    float percent = i * 100f / binLen;
                    OnEventReportProgress(percent);
                    if (Cancel) {
                        throw new Exception("Cancel: WriteMemory...");
                    }
                    RetryCntErrase = 0;
                    //Thread.Sleep(150);
                }
                catch (Exception ex) {
                    if (RetryCntErrase < 5) { //try again
                        OnEventReportMessage($"Retry '{RetryCntErrase}' ErasePage '{i}': {ex.Message}", true);
                        RetryCntErrase++;
                        i -= PageSize;
                        continue;
                    }
                    throw ex;
                }
            }
            OnEventReportProgress(100);
        }
        void EraseGlobal() {
            /* command word */
            var tx = new byte[4];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.ERASE;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* erase global page */
            tx[2] = 0xff;
            /* checksum */
            tx[3] = (byte)~ComputeChecksum(tx, 2, 2);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                //SerialWrite(tx, 1, 1);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                /* send address */
                SerialWrite(tx, 2, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Special Code Rejected");

                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
        void EraseSpecial(STExtendedEraseMode mode) {
            /* command word */
            var tx = new byte[5];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.EXT_ERASE;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* erase single page */
            tx[2] = (byte)((int)mode >> 8);
            tx[3] = (byte)((int)mode >> 0);
            /* checksum */
            tx[4] = (byte)~ComputeChecksum(tx, 2, 3);

            /* try to send command and wait for response */
            try {
                OnEventReportMessage("Global erase (20sec timeout)...", false);
                SerialWrite(tx, 0, 2);
                ///* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                /* send address */
                SerialWrite(tx, 2, 3);
                ///* wait for response code. use longer timeout, erase might
                // * take a while or two. */
                SerialRead(tmp, 0, 1, 20000);
                ///* check response code */
                if (tmp[0] != (byte)STResps.ACK) {
                    throw new Exception("Special code Rejected");
                }
                

                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
        public void ErasePage(uint pageNumber) {
            /* 'classic' erase operation supported? */
            if (Commands.Contains(STCmds.ERASE)) {
                Erase(pageNumber);
                return;
                /* 'extended' erase operation supported? */
            }
            else if (Commands.Contains(STCmds.EXT_ERASE)) {
                ExtendedErase(pageNumber);
                return;
                /* no operation supported */
            }
            else {
                throw new Exception("Command 'ErasePage' not supported");
            }
        }
        void Erase(uint pageNumber) {
            /* command word */
            var tx = new byte[5];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.ERASE;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* erase single page */
            tx[2] = 0;
            /* set page number */
            tx[3] = (byte)pageNumber;
            /* checksum */
            tx[4] = (byte)~ComputeChecksum(tx, 2, 2);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                ///* wait for response code */
                SerialRead(tmp, 0, 1);
                //tmp = SerialTransmission(tx, true, 5);
                if (tmp.Length == 0) {
                    throw new Exception("no Response");
                }
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Erase: Command Rejected");

                /* send address */
                SerialWrite(tx, 2, 3);
                ///* wait for response code */
                SerialRead(tmp, 0, 1);
                ///* check response code */
                if (tmp[0] != (byte)STResps.ACK) { 
                    throw new Exception("Erase: Page Rejected");
                }

                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }

            /* release semaphore */
            //sem.Release();
        }
        void ExtendedErase(uint pageNumber) {
            var tmp = new byte[1];
            /* command word */
            var tx = new byte[7];
            /* command code */
            tx[0] = (byte)STCmds.EXT_ERASE;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* erase single page */
            tx[2] = 0;
            tx[3] = 0;
            /* set page number */
            tx[4] = (byte)(pageNumber >> 8);
            tx[5] = (byte)(pageNumber >> 0);
            /* checksum */
            tx[6] = (byte)~ComputeChecksum(tx, 2, 5);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                if (tmp.Length == 0) {
                    throw new Exception("no Response");
                }
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("ExtendedErase: Command Rejected");

                /* send address */
                SerialWrite(tx, 2, 5);
                /* wait for response code. use longer timeout, erase might
                 * take a while or two. */
                SerialRead(tmp, 0, 1, 3000);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Page Rejected");

                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }

            /* release semaphore */
            //sem.Release();
        }
        public void StepWriteMemory() {
            if (FileBuffer == null) {
                throw new Exception("FileBuffer is empty. Load firmware file first.");
            }
            /* number of bytes written */
            int size = FileBuffer.Length;
            int bwritten = 0, btotal = size;
            int offset = 0;
            uint address = Address;
            Cancel = false;

            /* no support for read? */
            if (!Commands.Contains(STCmds.WRITE))
                throw new Exception("Command not supported");
            OnEventReportProgress(0);
            RetryCntWrite = 0;
            /* data is read in chunks */
            while (size > 0) {
                if (Cancel) {
                    throw new Exception("Cancel: WriteMemory...");
                }
                try {
                    if (bwritten == btotal) {
                        break;
                    }
                    /* chunk size */
                    int csize = Math.Min(size, 256);
                    /* read a single chunk */
                    Write(address, FileBuffer, offset, csize);

                    /* update iterators */
                    size -= csize; offset += csize; address += (uint)csize;
                    /* update number of bytes read */
                    bwritten += csize;
                    /* report progress */
                    float percent = (float)bwritten / (float)btotal * 100f;
                    OnEventReportProgress(percent);
                }
                catch (Exception ex) {
                    if (RetryCntWrite < 5) { //try again
                        OnEventReportMessage($"Retry '{RetryCntWrite}': {ex.Message}", true);
                        RetryCntWrite++;
                        continue;
                    }
                    HandleError(ex);
                    throw ex;
                }
            }
            //OnEventReportProgress(100);
        }
        void Write(uint address, byte[] data, int offset, int length) {
            var tx = new byte[9];
            var tmp = new byte[1];

            tx[0] = (byte)STCmds.WRITE;
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* store address */
            tx[2] = (byte)((address >> 24) & 0xff);
            tx[3] = (byte)((address >> 16) & 0xff);
            tx[4] = (byte)((address >> 8) & 0xff);
            tx[5] = (byte)(address & 0xff);
            /* address checksum (needs to be not negated. why? because ST! 
             * that's why. */
            tx[6] = (byte)~ComputeChecksum(tx, 2, 4);

            tx[7] = (byte)(length - 1);
            tx[8] = (byte)(~(ComputeChecksum(data, offset, length) ^ tx[7]));

            
            try {
                SerialWrite(tx, 0, 2);
                SerialRead(tmp, 0, 1);
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");
                SerialWrite(tx, 2, 5);
                SerialRead(tmp, 0, 1);
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Address Rejected");

                SerialWrite(tx, 7, 1);
                SerialWrite(data, offset, length);
                SerialWrite(tx, 8, 1);
                SerialRead(tmp, 0, 1);
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Data Rejected");
            }
            catch (Exception ex) {
                HandleError(ex);
                throw new Exception($"WriteData->{ex.Message}");
            }
        }
        void SerialWrite(byte[] data, int offset, int count) {
            SP.BaseStream.Write(data, offset, count);
        }
        void SerialRead(byte[] data, int offset, int count) {
            SerialRead(data, offset, count, 200);
        }
        void SerialRead(byte[] data, int offset, int count, int timeoutmSec) {
            int br = 0;
            SP.ReadTimeout = timeoutmSec;
            try {
                while (br < count) {
                    int reading = SP.ReadByte();
                    if (reading == -1) {
                        throw new Exception("end of stream");
                    }
                    data[br + offset] = (byte)reading;
                    br++;
                }
            }
            catch (Exception ex) {
                if (br == 1) {
                    if (data[0] == 31) {
                        throw new Exception("Timeout, Command rejected.");
                    }
                }
                throw new Exception($"SerialRead ({br}) from ({count}) Error: {ex.Message}");
            }
        }

        public void ReadMemory(int size) {
            FileBufferRead = new byte[size];
            int offset = 0;
            Cancel = false;
            int bread = 0;
            uint address = Address;
            /* no support for read? */
            if (!Commands.Contains(STCmds.READ))
                throw new Exception("Command not supported");
            OnEventReportProgress(0);
            RetryCntWrite = 0;
            /* data is read in chunks */
            //int sizeLeft = size;
            while (bread < size) {
                if (Cancel) {
                    throw new Exception("Cancel: ReadMemory...");
                }
                try {
                    /* chunk size */
                    int csize = Math.Min(size - bread, 256);
                    /* read a single chunk */
                    Read(address, FileBufferRead, offset, csize);

                    /* update iterators */
                    offset += csize; address += (uint)csize;
                    /* update number of bytes read */
                    bread += csize;
                    float percent = (float)bread / (float)size * 100f;
                    OnEventReportProgress(percent);
                    RetryCntWrite = 0;
                    continue;
                }
                catch (Exception ex) {
                    if (RetryCntWrite < 5) { //try again
                        OnEventReportMessage($"Retry '{RetryCntWrite}': {ex.Message}", true);
                        RetryCntWrite++;
                        if (RetryCntWrite>3) {
                            StepInit();
                        }
                        continue;
                    }
                    HandleError(ex);
                    throw ex;
                }
            }
            OnEventReportProgress(100);
        }
        public void StepWriteBinary(string filePath) {
            if (File.Exists(filePath)) {
                File.Delete(filePath);
            }
            FileStream fs = new FileStream(filePath, FileMode.CreateNew);
            fs.Write(FileBufferRead, 0, FileBufferRead.Length);
            fs.Close();
        }
        void Read(uint address, byte[] buf, int offset, int length) {
            /* command word */
            var tx = new byte[9];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.READ;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* store address */
            tx[2] = (byte)((address >> 24) & 0xff);
            tx[3] = (byte)((address >> 16) & 0xff);
            tx[4] = (byte)((address >> 8) & 0xff);
            tx[5] = (byte)(address & 0xff);
            /* address checksum (needs to be not negated. why? because ST! 
             * that's why. */
            tx[6] = (byte)~ComputeChecksum(tx, 2, 4);

            /* store number of bytes */
            tx[7] = (byte)(length - 1);
            /* size checksum */
            tx[8] = ComputeChecksum(tx, 7, 1);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                /* send address */
                SerialWrite(tx, 2, 5);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Address Rejected");

                /* send size */
                SerialWrite(tx, 7, 1);
                SerialWrite(tx, 8, 1);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Size Rejected");

                /* receive response */
                SerialRead(buf, offset, length);
                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
        public void WriteUnprotect() {
            /* command word */
            var tx = new byte[2];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.WR_UNPROTECT;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                /* wait for response code. use longer timeout, erase might
                 * take a while or two. */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Write Unprotect Rejected");

                /* oops, something baaad happened! */
            }
            catch(Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
        public void ReadUnprotect() {
            /* command word */
            var tx = new byte[2];
            /* temporary storage for response bytes */
            var tmp = new byte[1];

            /* command code */
            tx[0] = (byte)STCmds.RD_UNPROTECT;
            /* checksum */
            tx[1] = ComputeChecksum(tx, 0, 1);

            /* try to send command and wait for response */
            try {
                /* send bytes */
                SerialWrite(tx, 0, 2);
                /* wait for response code */
                SerialRead(tmp, 0, 1);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Command Rejected");

                /* wait for response code. use longer timeout, erase might
                 * take a while or two. */
                SerialRead(tmp, 0, 10000);
                /* check response code */
                if (tmp[0] != (byte)STResps.ACK)
                    throw new Exception("Write Unprotect Rejected");

                /* oops, something baaad happened! */
            }
            catch (Exception ex) {
                HandleError(ex);
                throw ex;
            }
        }
    }
}
